module net.BurtonRadons.dig.gl;

private import net.BurtonRadons.dig.platform.base;
private import net.BurtonRadons.dig.common.math;

import net.BurtonRadons.dig.platform.canvasGL;

/** A buffered light. */
class GLlight
{
public:
    GLenum index; /**< LIGHTi value. */

    /** Set the parameters. */
    this (GLenum index) { this.index = index; }

    /** Set the ambient color. */
    void ambient (GLfloat r, GLfloat g, GLfloat b) { gl.lightAmbient (index, r, g, b); }

    /** Set the attenuation parameters. */
    void attenuation (GLfloat constant, GLfloat linear, GLfloat quadratic) { gl.lightAttenuation (index, constant, linear, quadratic); }

    /** Set the constant attenuation. */
    void constantAttenuation (GLfloat value) { gl.lightConstantAttenuation (index, value); }

    /** Set the diffuse color. */
    void diffuse (GLfloat r, GLfloat g, GLfloat b) { gl.lightDiffuse (index, r, g, b); }

    /** Disable this light. */
    void disable () { gl.disable (index); }

    /** Enable this light. */
    void enable () { gl.enable (index); }

    /** Get whether this light is enabled. */
    GLboolean isEnabled () { return gl.isEnabled (index); }

    /** Set the linear attenuation. */
    void linearAttenuation (GLfloat value) { gl.lightLinearAttenuation (index, value); }

    /** Set the position of the light. */
    void position (GLfloat x, GLfloat y, GLfloat z) { gl.lightPosition (index, x, y, z); }

    /** Set the quadratic attenuation. */
    void quadraticAttenuation (GLfloat value) { gl.lightQuadraticAttenuation (index, value); }

    /** Set the specular color. */
    void specular (GLfloat r, GLfloat g, GLfloat b) { gl.lightSpecular (index, r, g, b); }

    /** Set the normal of the spotlight. */
    void spotDirection (GLfloat x, GLfloat y, GLfloat z) { gl.lightSpotDirection (index, x, y, z); }

    /** Set the cutoff angle of the spotlight (0 to 90 degrees, or 180 degrees (default) for a normal light). */
    void spotCutoff (GLfloat angle) { gl.lightSpotCutoff (index, angle); }

    /** Set the normal of the spotlight. */
    void spotDirection (vec3 v) { gl.lightSpotDirection (index, v); }

    /** Set the intensity distribution of the spotlight (0 for diffuse, 128 for hard-edged). */
    void spotExponent (GLfloat factor) { gl.lightSpotExponent (index, factor); }
};

/** A texture object.  This wraps a texture name in a convenient bow and
    package for GL, and mirrors an API there that starts entirely with
    texture.  You create a texture object using the #textureNew method on
    the frame.

    Modifying and polling texture state does not require binding beforehand
    and will not change your binding afterwards.
  */

class GLtexture
{
    const static GLenum target = GL.TEXTURE_2D; /**< Target type. */
    const static GLenum target_binding = GL.TEXTURE_2D_BINDING; /**< Target binding. */

    GLuint name; /**< Name of this texture. */
    bit owner = true; /**< If owned, it can delete the texture name when deleted. */

    /** Set the frame and allocate the name. */
    this ()
    {
        this.name = gl.textureCreate ();
        this.owner = true;
    }

    /** Delete the texture name if @a owner is set. */
    ~this ()
    {
        if (owner)
            gl.textureDelete (name);
    }

/+
#ifndef DOXYGEN_SHOULD_SKIP_THIS
+/
final:
/+
#endif
+/

    /** Bind the texture.  This makes it current GL state and will be used
      * for texturing if #enable (<b>TEXTURE_2D</b>).
      */

    void bind ()
    {
        gl.textureBind (name);
    }

    /** Copy a rectangle of the color buffer into the texture,
      * replacing any data which was in there before.  level is the
      * mipmapping level to read from.  format is the texture format
      * to store and can be <b>ALPHA</b>, <b>LUMINANCE</b>, 
      * <b>LUMINANCE_ALPHA</b>, <b>INTENSITY</b>, <b>RGB</b>, <b>RGBA</b>,
      * or a number of more obscure modes.  x and y are the starting
      * coordinates for the color buffer rectangle; a y of zero is the
      * bottom row of the buffer.  width and height are the dimensions
      * of the rectangle and must be powers of two.
      */

    void copyImage (GLint level, GLenum format, GLint x, GLint y, GLint width, GLint height)
    {
        GLuint old;
        
        store (old);
        gl.textureCopyImage (level, format, x, y, width, height);
        restore (old);
    }

    /** Enable texturing and bind this texture. */
    void enable ()
    {
        gl.enable (target);
        gl.textureBind (name);
    }

    /** Disable texturing. */
    void disable ()
    {
        gl.disable (target);
    }

    /** Set whether to automatically generate mipmaps when texture data is loaded.
      * This is dependent on the SGIS_generate_mipmap extension, so it returns
      * whether the extension was available.  "value" is whether to enable mipmap
      * generation.
      */

    GLboolean generateMipmap (GLboolean value)
    {
        GLuint old;
        GLboolean result;

        store (old);
        result = gl.textureGenerateMipmap (value);
        restore (old);
        return result;
    }

    /** Get the height of the texture in pixels. */
    GLint height ()
    {
        return height (0);
    }

    /** Get the height of the texture in pixels at the specific mipmap level. */
    GLint height (GLint level)
    {
        GLuint old;
        GLint result;

        store (old);
        result = gl.textureHeight (level);
        restore (old);
        return result;
    }

    /** Upload image into the texture.
      * @a width and @a height are the dimensions of the texture in pixels.
      * @a format is the provided channels and can be <b>COLOR_INDEX</b>,
      * <b>RED</b>, <b>GREEN</b>, <b>BLUE</b>, <b>ALPHA</b>,
      * <b>RGB</b>, <b>RGBA</b>, <b>LUMINANCE</b>, and <b>LUMINANCE_ALPHA</b>.
      * @a type is the data type and can be <b>UNSIGNED_BYTE</b> (GLubyte),
      * <b>BYTE</b> (GLbyte), <b>BITMAP</b> (compressed bits, @a format must be <b>COLOR_INDEX</b>),
      * <b>UNSIGNED_SHORT</b> (GLushort), <b>SHORT</b> (GLshort),
      * <b>UNSIGNED_INT</b> (GLuint), <b>INT</b> (GLint), and
      * <b>FLOAT</b> (GLfloat).  @a pixels is the array to read from;
      * if null, the texture dimensions are set but are not cleared.
      *
      * The image is loaded from the bottom up, so the first pixel
      * in the @a pixels array is the bottom-left one.
      *
      * If the width and height are not powers of two (1, 2, 4, 8, 16, 32,
      * 64, 128, 256, 512, 1024, 2048, 4096, or 8192), the image is first
      * resized to the nearest demarcation.  Then it is checked whether
      * it fits, and is progressively halved in size until it does.  Mipmaps
      * are automatically generated, either manually or by extension.
      */

    void image (GLint width, GLint height, GLenum format, GLenum type, GLvoid *data)
    {
        GLuint old;
        
        store (old);
        gl.textureImage (width, height, format, type, data);
        restore (old);
    }

    /** Read the texture image and store it in data. */
    void image (GLenum format, out GLubyte [] pixels)
    {
        GLuint old;

        store (old);
        gl.textureImage (0, format, pixels);
        restore (old);
    }

    /** Read the texture image at a specific level and store it in data. */
    void image (GLint level, GLenum format, out GLubyte [] pixels)
    {
        GLuint old;

        store (old);
        gl.textureImage (level, format, pixels);
        restore (old);
    }

    /** Set the magnification filter to either <b>NEAREST</b> or <b>LINEAR</b>.
        The magnification filter is used when the pixel is smaller than a
        texture element.  It can be one of:

        - <b>NEAREST</b>: Returns the value of the texture element that is nearest (in Manhattan distance) to the center of the pixel being textured.
        - <b>LINEAR</b>: Returns the weighted average of the four texture elements that are closest to the center of the pixel being textured. These can include border texture elements, depending on the values of #textureWrapS and #textureWrapT, and on the exact mapping.
      */

    void magFilter (GLenum mode)
    {
        GLuint old;
        
        store (old);
        gl.textureMagFilter (mode);
        restore (old);
    }

    /** Set the minification filter to one of
        <b>NEAREST</b>, <b>LINEAR</b>, <b>NEAREST_MIPMAP_NEAREST</b>,
        <b>LINEAR_MIPMAP_NEAREST</b>, <b>NEAREST_MIPMAP_LINEAR</b>, or
        <b>LINEAR_MIPMAP_LINEAR</b>).
  
        The minification filter is used when the pixel is larger than a
        texture element.  It can be one of:

        - <b>NEAREST</b>: Returns the value of the texture element that is nearest (in Manhattan distance) to the center of the pixel being textured.
        - <b>LINEAR</b>: Returns the weighted average of the four texture elements that are closest to the center of the pixel being textured. These can include border texture elements, depending on the values of #textureWrapS and #textureWrapT, and on the exact mapping.
        - <b>NEAREST_MIPMAP_NEAREST</b>: Chooses the mipmap that most closely matches the size of the pixel being textured and uses the <b>NEAREST</b> criterion (the texture element nearest to the center of the pixel) to produce a texture value.
        - <b>LINEAR_MIPMAP_NEAREST</b>: Chooses the mipmap that most closely matches the size of the pixel being textured and uses the <b>LINEAR</b> criterion (a weighted average of the four texture elements that are closest to the center of the pixel) to produce a texture value.
        - <b>NEAREST_MIPMAP_LINEAR</b>: Chooses the two mipmaps that most closely match the size of the pixel being textured and uses the <b>NEAREST</b> criterion (the texture element nearest to the center of the pixel) to produce a texture value from each mipmap. The final texture value is a weighted average of those two values.
        - <b>LINEAR_MIPMAP_LINEAR</b>: Chooses the two mipmaps that most closely match the size of the pixel being textured and uses the <b>LINEAR</b> criterion (a weighted average of the four texture elements that are closest to the center of the pixel) to produce a texture value from each mipmap.  The final texture value is a weighted average of those two values.     

        <b>LINEAR</b> is often called bilinear filtering, while
        <b>LINEAR_MIPMAP_LINEAR</b> is often called trilinear filtering.
        <b>NEAREST</b> and <b>LINEAR</b> are often slower than the
        other forms due to what's called cache thrashing: too much access to
        widely spaced sections of memory.
     */

    void minFilter (GLenum mode)
    {
        GLuint old;
        
        store (old);
        gl.textureMinFilter (mode);
        restore (old);
    }

    /** Set the texture's residence priority. */
    void priority (GLfloat value)
    {
        GLuint old;

        store (old);
        gl.texturePriority (value);
        restore (old);
    }

    /** Get the texture's residence priority. */
    GLfloat priority ()
    {
        GLuint old;
        GLfloat result;

        store (old);
        result = gl.texturePriority ();
        restore (old);
        return result;
    }

    /** Get the width of the texture in pixels. */
    GLint width ()
    {
        return width (0);
    }

    /** Get the width of the texture in pixels at the specific mipmap level. */
    GLint width (GLint level)
    {
        GLuint old;
        GLint result;

        store (old);
        result = gl.textureWidth (level);
        restore (old);
        return result;
    }

    /** Sets the wrap parameter for texture coordinate @a s
        to either <b>CLAMP</b> or <b>REPEAT</b>.  <b>CLAMP</b> causes
        @a s coordinates to be  clamped to the range [0, 1] and
        is useful for preventing wrapping artifacts when
        mapping a std.math.single image onto an object. <b>REPEAT</b>
        causes the integer part of the @a s coordinate  to be
        ignored; the GL uses only the fractional part,
        thereby creating a repeating pattern. Border
        texture elements are accessed only if wrapping is
        set to <b>CLAMP</b>.  Initially, @a wrapS is
        set to <b>REPEAT</b>.
      */

    void wrapS (GLenum value)
    {
        GLuint old;

        store (old);
        gl.textureWrapS (value);
        restore (old);
    }

    /** Get the wrap parameter for texture coordinate @a s. */
    GLenum wrapS ()
    {
        GLuint old;
        GLenum result;

        store (old);
        result = gl.textureWrapS ();
        restore (old);
        return result;
    }

    /** Sets the wrap parameter for texture coordinate @a t
        to either <b>CLAMP</b> or <b>REPEAT</b>.  <b>CLAMP</b> causes
        @a t coordinates to be clamped to the range [0, 1] and
        is useful for preventing wrapping artifacts when
        mapping a std.math.single image onto an object. <b>REPEAT</b>
        causes the integer part of the @a t coordinate to be
        ignored; the GL uses only the fractional part,
        thereby creating a repeating pattern. Border
        texture elements are accessed only if wrapping is
        set to <b>CLAMP</b>.  Initially, @a wrapT is
        set to <b>REPEAT</b>.
      */

    void wrapT (GLenum value)
    {
        GLuint old;

        store (old);
        gl.textureWrapT (value);
        restore (old);
    }

    /** Get the wrap parameter for texture coordinate @a t. */
    GLenum wrapT ()
    {
        GLuint old;
        GLenum result;

        store (old);
        result = gl.textureWrapT ();
        restore (old);
        return result;
    }

/+
#ifdef DoxygenShouldSkipThis
+/

    void store (out GLuint old)
    {
        old = gl.getInteger (target_binding);
        if (old != name)
            gl.textureBind (name);
    }

    void restore (GLuint old)
    {
        if (old != name)
            gl.textureBind (old);
    }
    
/+
#endif
+/
}

/** A display list object.  This wraps a display list name with a
  * pretty bow and reflects the GL API for display lists (all
  * such methods start with "list").
  */

class GLlist
{
public:
    GLuint name; /**< Name of this display list. */
    bit owner = true; /**< If owned, it can delete the display list when deleted. */

    /** Set the frame and allocate the name. */
    this ()
    {
        this.name = gl.listCreate ();
        this.owner = true;
    }

    /** Delete the texture name if @a owner is set. */
    ~this ()
    {
        if (owner)
            gl.listDelete (name);
    }

    /** Start compilation on the list, deleting any previous data. */
    void compile ()
    {
        gl.listCompile (name);
    }

    /** Start compilation on the list and execute at the same time. */
    void compileAndExecute ()
    {
        gl.listCompileAndExecute (name);
    }

    /** Finish compilation. */
    void compileEnd ()
    {
        gl.listCompileEnd ();
    }

    /** Run the display list. */
    void call ()
    {
        gl.listCall (name);
    }
};

/** The base OpenGL wrapper class, with a std.math.singleton in gl. */
class GL
{
/+
#ifndef DOXYGEN_SHOULD_SKIP_THIS
+/
    CanvasGL digCommonCurrentCanvas;
    
final:
/+
#endif
+/

    /** The error class thrown by GL.  Most OpenGL errors keep it in
      * defined state, with the exception of OUT_OF_MEMORY.
      */

    class GLError : Error
    {
        GLenum code; /**< The GL error code being thrown. */
    
        /** Sets the error code and std.string. */
        this (GLenum code)
        {
            super (codeString (code));
            this.code = code;
        }
    
        /** Get the string for a code. */
        static char [] codeString (GLenum code)
        {
            switch (code)
            {
            case NO_ERROR: return "OpenGL: No error";
            case INVALID_ENUM: return "OpenGL: Invalid enumeration; an unacceptable value is specified for an enumerated argument";
            case INVALID_VALUE: return "OpenGL: Invalid value; a numeric argument is out of range";
            case INVALID_OPERATION: return "OpenGL: Invalid operation; the specified operation is not allowed in the current state";
            case STACK_OVERFLOW: return "OpenGL: Stack overflow";
            case STACK_UNDERFLOW: return "OpenGL: Stack underflow";
            case OUT_OF_MEMORY: return "OpenGL: Out of memory; there is not enough memory to execute the command; GL state is undefined!";
            }
        }
    }

    /** Assign the alpha test function parameters which is enabled with #enable (<b>ALPHA_TEST</b>).
      * When enabled, fragments which fail the test, which is between the fragment's alpha and a
      * reference value, will not be rendered at all.  This is most often used to give a sharp line to
      * vegetation and cloth.  The possible values for test are, where alpha is the fragment's alpha
      * and ref is the reference value passed to this function:
      *
      * - <b>NEVER</b>: Never passes (false).
      * - <b>LESS</b>: alpha < ref.
      * - <b>EQUAL</b>: alpha == ref.
      * - <b>LEQUAL</b>: alpha <= ref.
      * - <b>GREATER</b>: alpha > ref.
      * - <b>NOTEQUAL</b>: alpha != ref.
      * - <b>GEQUAL</b>: alpha >= ref.
      * - <b>ALWAYS</b>: Always passes (true).
      *
      * The default is (<b>ALWAYS</b>, 0).
      */

    void alphaFunc (GLenum func, GLclampf ref)
    {
        glAlphaFunc (func, ref);
    }

    /** Start the vertices of a primitive or a group of like primitives, terminated with end
      * (mode is one of <b>POINTS</b>, <b>LINES</b>, <b>LINE_STRIP</b>,
      * <b>LINE_LOOP</b>, <b>TRIANGLES</b>, <b>TRIANGLE_STRIP</b>,
      * <b>TRIANGLE_FAN</b>, <b>QUADS</b>, <b>QUAD_STRIP</b>, or
      * <b>POLYGON</b>.)
      */

    void begin (GLenum mode)
    { 
        glBegin (mode);
    }

    /** Specify the blend equation.  This determines how the two colors that
      * result from #blendFunc are combined into one.  The possible values are,
      * where S is the source color and D is the destination color:
      *
      * - <b>FUNC_ADD</b> (default): S + D.
      * - <b>FUNC_MIN</b>: min (S, D).
      * - <b>FUNC_MAX</b>: max (S, D).
      * - <b>FUNC_SUBTRACT</b>: S - D.
      * - <b>FUNC_REVERSE_SUBTRACT</b>: D - S.
      *
      * @a blendEquation is not supported on all hardware.  If it isn't,
      * writing to it has no effect on the value.
      */

    void blendEquation (GLenum mode) 
    { 
        if (digCommonCurrentCanvas.glBlendEquationEXT) 
            digCommonCurrentCanvas.glBlendEquationEXT (mode); 
    }

    /** Read the blend equation. */
    GLenum blendEquation () 
    { 
        return getInteger (BLEND_EQUATION_EXT); 
    }

    /** Specify the pixel arithmetic.  The sfactor is how the source blending
      * color is computed; the source color is the fragment being drawn.  The
      * dfactor is how the destination blending color is computed; the
      * destination color is the color buffer value.  How these results are
      * combined to form the output color depends upon the setting of
      * #blendEquation.
      *
      * In the following list of allowable @a sfactor and @a dfactor values,
      * I give the color the parameter is multiplied against.  S is the source
      * color (Sr, Sg, Sb, and Sa its components), D is the destination color.
      *
      * - <b>ZERO</b>: (0, 0, 0, 0).
      * - <b>ONE</b>: (1, 1, 1, 1).
      * - <b>SRC_COLOR</b>: S (dfactor only).
      * - <b>ONE_MINUS_SRC_COLOR</b> (1 - Sr, 1 - Sg, 1 - Sb, 1 - Sa) (dfactor only).
      * - <b>DST_COLOR</b>: D (sfactor only).
      * - <b>ONE_MINUS_DST_COLOR</b> (1 - Sr, 1 - Sg, 1 - Sb, 1 - Sa) (sfactor only).
      * - <b>SRC_ALPHA</b>: (Sa, Sa, Sa, Sa).
      * - <b>ONE_MINUS_SRC_ALPHA</b>: (1 - Sa, 1 - Sa, 1 - Sa, 1 - Sa).
      * - <b>DST_ALPHA</b>: (Da, Da, Da, Da).
      * - <b>ONE_MINUS_DST_ALPHA</b>: (1 - Da, 1 - Da, 1 - Da, 1 - Da).
      * - <b>SRC_ALPHA_SATURATE</b>: Where (f = min (Sa, 1 - Da)), (f, f, f, 1).
      *
      * For this to have any effect, #enable (BLEND) must be set.
      */

    void blendFunc (GLenum sfactor, GLenum dfactor) 
    { 
        glBlendFunc (sfactor, dfactor); 
    }

    /** Clear buffers to preset values (mask is a combination of COLOR_BUFFER_BIT,
      * DEPTH_BUFFER_BIT, ACCUM_BUFFER_BIT, and STENCIL_BUFFER_BIT.)
      */

    void clear (GLbitfield mask) 
    { 
        glClear (mask); 
    }

    /** Set the clear color used when calling clear (COLOR_BUFFER_BIT). */
    void clearColor (GLfloat r, GLfloat g, GLfloat b) 
    { 
        clearColor (r, g, b, 0); 
    }

    /** Set the clear color used when calling clear (COLOR_BUFFER_BIT). */
    void clearColor (GLfloat r, GLfloat g, GLfloat b, GLfloat a) 
    { 
        glClearColor (r, g, b, a); 
    }

    /** Set the clear color used when calling clear (COLOR_BUFFER_BIT). */
    void clearColor (Color c) 
    { 
        glClearColor (c.r / 255.0, c.g / 255.0, c.b / 255.0, c.a / 255.0); 
    }

    /** Get the clear color used when calling clear (COLOR_BUFFER_BIT). */
    Color clearColor () 
    { 
        GLfloat [4] color; 
    
        glGetFloatv (COLOR_CLEAR_VALUE, color); 
    
        return AColor (color [0] * 255, color [1] * 255, color [2] * 255, color [3] * 255); 
    }

    /** Set the clear depth used when calling clear (DETH_BUFFER_BIT).
      * The initial value is 1, and input is clamped to the range [0, 1].
      */

    void clearDepth (GLclampd depth) 
    { 
        glClearDepth (depth); 
    }

    /** Get the clear depth. */
    GLdouble clearDepth () 
    { 
        return getDouble (DEPTH_CLEAR_VALUE); 
    }

    /** Set the color of the next vertex. */
    void color (GLfloat r, GLfloat g, GLfloat b) 
    { 
        glColor3f (r, g, b); 
    }

    /** Set the color of the next vertex. */
    void color (GLfloat r, GLfloat g, GLfloat b, GLfloat a) 
    { 
        glColor4f (r, g, b, a); 
    }

    /** Set the color of the next vertex. */
    void color (Color c) 
    { 
        glColor4ub (c.r, c.g, c.b, c.a); 
    }

    /** Set the color array; use #enable (<b>COLOR_ARRAY</b>) to enable it.
      * #a stride should be zero to indicate 4 (the size of the cell) or
      * the number of bytes between each entry.
      */

    void colorArray (Color *array, int stride)
    {
        glColorPointer (4, UNSIGNED_BYTE, stride, array);
    }

    /** Enable or disable writing of frame buffer components.
      * Disabling a component means that it will not be modified by
      * any drawing operation.
      */

    void colorMask (GLboolean red, GLboolean green, GLboolean blue, GLboolean alpha) 
    { 
        glColorMask (red, green, blue, alpha); 
    }

    /** Enable or disable writing of the red frame buffer component. */
    void colorMaskRed (GLboolean value) 
    { 
        glColorMask (value, colorMaskGreen (), colorMaskBlue (), colorMaskAlpha ()); 
    }

    /** Get whether the red frame buffer component is enabled. */
    GLboolean colorMaskRed () 
    { 
        GLboolean [4] b; 
    
        glGetBooleanv (COLOR_WRITEMASK, b); 
        return b [0]; 
    }

    /** Enable or disable writing of the green frame buffer component. */
    void colorMaskGreen (GLboolean value) 
    { 
        glColorMask (colorMaskRed (), value, colorMaskBlue (), colorMaskAlpha ()); 
    }

    /** Get whether the green frame buffer component is enabled. */
    GLboolean colorMaskGreen () 
    { 
        GLboolean [4] b; 
    
        glGetBooleanv (COLOR_WRITEMASK, b); 
        return b [1]; 
    }

    /** Enable or disable writing of the blue frame buffer component. */
    void colorMaskBlue (GLboolean value) 
    { 
        glColorMask (colorMaskRed (), colorMaskGreen (), value, colorMaskAlpha ()); 
    }

    /** Get whether the blue frame buffer component is enabled. */
    GLboolean colorMaskBlue () 
    { 
        GLboolean [4] b; 
    
        glGetBooleanv (COLOR_WRITEMASK, b); 
        return b [2]; 
    }

    /** Enable or disable writing of the alpha frame buffer component. */
    void colorMaskAlpha (GLboolean value) 
    { 
        glColorMask (colorMaskRed (), colorMaskGreen (), colorMaskBlue (), value); 
    }

    /** Get whether the alpha frame buffer component is enabled. */
    GLboolean colorMaskAlpha () 
    { 
        GLboolean [4] b; 
    
        glGetBooleanv (COLOR_WRITEMASK, b); 
        return b [3]; 
    }

    /** Set the current culling face to use when #enable (<b>CULL_FACE</b>) (one of <b>FRONT</b>, <b>BACK</b>, or <b>FRONT_AND_BACK</b>). */
    void cullFace (GLenum mode)
    { 
        glCullFace (mode); 
    }

    /** Get the current culling face. */
    GLenum cullFace () 
    { 
        return getInteger (CULL_FACE_MODE); 
    }

    /** Set the depth comparison function (@a func is one of <b>NEVER</b>, <b>LESS</b>,
        <b>EQUAL</b>, <b>LEQUAL</b>, <b>GREATER</b>, <b>NOTEQUAL</b>, <b>GEQUAL</b>, or <b>ALWAYS</b>).
    
        This determines whether a fragment wins the contest with the current depth buffer point. @a func is one of:

        - <b>NEVER</b>: Never passes.
        - <b>LESS</b> (default): Passes if the incoming depth value is less than the stored depth value.
        - <b>EQUAL</b>: Passes if the incoming depth value is equal to the stored depth value.
        - <b>LEQUAL</b>: Passes if the incoming depth value is less than or equal to the stored depth value.
        - <b>GREATER</b>: Passes if the incoming depth value is greater than the stored depth value.
        - <b>NOTEQUAL</b>: Passes if the incoming depth value is not equal to the stored depth value.
        - <b>GEQUAL</b>: Passes if the incoming depth value is greater than or equal to the stored depth value.
        - <b>ALWAYS</b>: Always passes.
      */

    void depthFunc (GLenum func) 
    { 
        glDepthFunc (func); 
    }

    /** Get the current depth comparison function. */
    GLenum depthFunc () 
    { 
        return getInteger (DEPTH_FUNC); 
    }

    /** Specify mapping of depth values from normalized device coordinates to window coordinates.
      * The initial parameters are 0 and 1, indicating full utilisation of the
      * depth buffer.  It is not necessary for zNear to be below zFar.
      */

    void depthRange (GLclampd zNear, GLclampd zFar) 
    { 
        glDepthRange (zNear, zFar); 
    }

    /** Disable a server-side capability. See #enable. */
    void disable (GLenum cap)
    {
        switch (cap)
        {
            case COLOR_ARRAY: glDisableClientState (cap); break;
            case DEPTH_WRITEMASK: glDepthMask (false); break;
            case EDGE_FLAG_ARRAY: glDisableClientState (cap); break;
            case INDEX_ARRAY: glDisableClientState (cap); break;
            case LIGHT_MODEL_LOCAL_VIEWER: glLightModeli (cap, FALSE); break;
            case LIGHT_MODEL_TWO_SIDE: glLightModeli (cap, FALSE); break;
            case NORMAL_ARRAY: glDisableClientState (cap); break;
            case SEPARATE_SPECULAR_COLOR: glLightModeli (LIGHT_MODEL_COLOR_CONTROL_EXT, SINGLE_COLOR_EXT); break;
            case TEXEL_ARRAY: glDisableClientState (cap); break;
            case VERTEX_ARRAY: glDisableClientState (cap); break;
            default: glDisable (cap); break;
        }
    }

    /** Disable a server-side capability and return the previous value. */
    bit disableAndGet (GLenum cap)
    {
        bit old = isEnabled (cap);

        disable (cap);
        return old;
    }

    /** Draw arrays according to the currently enabled arrays:
      *
      * - #enable (<b>NORMAL_ARRAY</b>), using #normalArray.
      * - #enable (<b>TEXEL_ARRAY</b>), using #texelArray.
      * - #enable (<b>VERTEX_ARRAY</b>), using #vertexArray.
      * - #enable (<b>COLOR_ARRAY</b>), using #colorArray.
      */

    void drawArrays (GLenum mode, GLint first, GLsizei count)
    {
        glDrawArrays (mode, first, count);
    }

    /** Draw an indexed set of elements.  This draws a std.math.single primitive
      * based on the currently enabled arrays (see #drawArrays for a
      * summary).  mode is the type of primitive to draw, and array is
      * the indices into the enabled arrays.
      */

    void drawElements (GLenum mode, GLuint [] array)
    {
        glDrawElements (mode, array.length, GL.UNSIGNED_INT, (void *) array);
    }

    /** Enable a server-side capability. 
      * @a cap is one of:

        - <b>ALPHA_TEST</b>: If enabled, do alpha testing. See #alphaFunc.
        - <b>AUTO_NORMAL</b>: If enabled, generate normal vectors when either <b>MAP2_VERTEX_3</b> or <b>MAP2_VERTEX_4</b> is used to generate vertices. See map2.
        - <b>BLEND</b>: If enabled, blend the incoming RGBA color values with the values in the color buffers. See #blendFunc.
        - <b>CLIP_PLANEi</b>: If enabled, clip geometry against user-defined clipping plane i.  See #clipPlane.
        - <b>COLOR_LOGIC_OP</b>: If enabled, apply the currently selected logical operation to the incoming RGBA color and color buffer values. See #logicOp.
        - <b>COLOR_MATERIAL</b>: If enabled, have one or more material parameters track the current color.  See #colorMaterial.
        - <b>CULL_FACE</b>: If enabled, cull polygons based on their winding in window coordinates. See #cullFace.
        - <b>DEPTH_TEST</b>: If enabled, do depth comparisons and update the depth buffer. Note that even if the depth buffer exists and the depth mask is non-zero, the depth buffer is not updated if the depth test is disabled. See #depthFunc and #depthRange.
        - <b>DEPTH_WRITEMASK</b>: If enabled (the default) and <b>DEPTH_TEST</b> is also enabled, write to the depth buffer.
        - <b>DITHER</b>: If enabled, dither color components or indices before they are written to the color buffer.
        - <b>FOG</b>: If enabled, blend a fog color into the posttexturing color.  See #fogMode, #fogDensity, #fogDistanceMode, #fogStart, #fogEnd, #fogIndex, and #fogColor.
        - <b>INDEX_LOGIC_OP</b>: If enabled, apply the currently selected logical operation to the incoming index and color buffer indices. See #logicOp.
        - <b>LIGHT_MODEL_LOCAL_VIEWER</b>: If disabled, specular reflection angles take the view direction to be parallel to and in the direction of the -z axis, regardless of the location of the vertex in eye coordinates.  Otherwise, specular reflections are computed from the origin of the eye coordinate system.
        - <b>LIGHT_MODEL_TWO_SIDE</b>: If enabled, use the <b>BACK</b> material parameters when the polygon is back-facing the lights.
        - <b>LIGHTi</b>: If enabled, include light i in the evaluation of the lighting equation. See #lightModel and #light.
        - <b>LIGHTING</b>: If enabled, use the current lighting parameters to compute the vertex color or index.  Otherwise, simply associate the current color or index with each vertex. See #material, #lightModel, and #light.
        - <b>LINE_SMOOTH</b>: If enabled, draw lines with correct filtering.  Otherwise, draw aliased lines.  See #lineWidth.
        - <b>LINE_STIPPLE</b>: If enabled, use the current line stipple pattern when drawing lines. See #lineStipple.
        - <b>MAP1_COLOR_4</b>: If enabled, calls to #evalCoord1, #evalMesh1, and #evalPoint1 generate RGBA values.  See #map1.
        - <b>MAP1_INDEX</b>: If enabled, calls to #evalCoord1, #evalMesh1, and #evalPoint1 generate color indices.  See #map1.
        - <b>MAP1_NORMAL</b>: If enabled, calls to #evalCoord1, #evalMesh1, and #evalPoint1 generate normals.  See #map1.
        - <b>MAP1_TEXTURE_COORD_1</b>: If enabled, calls to #evalCoord1, #evalMesh1, and #evalPoint1 generate @a s texture coordinates. See #map1.
        - <b>MAP1_TEXTURE_COORD_2</b>: If enabled, calls to #evalCoord1, #evalMesh1, and #evalPoint1 generate @a s and @a t texture coordinates.  See #map1.
        - <b>MAP1_TEXTURE_COORD_3</b>: If enabled, calls to #evalCoord1, #evalMesh1, and #evalPoint1 generate @a s, @a t, and @a r texture coordinates.  See #map1.
        - <b>MAP1_TEXTURE_COORD_4</b>: If enabled, calls to #valCoord1, #evalMesh1, and #evalPoint1 generate @a s, @a t, @a r, and @a q texture coordinates.  See #map1.
        - <b>MAP1_VERTEX_3</b>: If enabled, calls to #evalCoord1, #evalMesh1, and #evalPoint1 generate @a x, @a y, and @a z vertex coordinates.  See #map1.
        - <b>MAP1_VERTEX_4</b>: If enabled, calls to #evalCoord1, #evalMesh1, and #evalPoint1 generate homogeneous @a x, @a y, @a z, and @a w vertex coordinates.  See #map1.
        - <b>MAP2_COLOR_4</b>: If enabled, calls to #evalCoord2, #evalMesh2, and #evalPoint2 generate RGBA values.  See #map2.
        - <b>MAP2_INDEX</b>: If enabled, calls to #evalCoord2, #evalMesh2, and #evalPoint2 generate color indices.  See #map2.
        - <b>MAP2_NORMAL</b>: If enabled, calls to #evalCoord2, #evalMesh2, and #evalPoint2 generate normals.  See #map2.
        - <b>MAP2_TEXTURE_COORD_1</b>: If enabled, calls to #evalCoord2, #evalMesh2, and #evalPoint2 generate @a s texture coordinates. See #map2.
        - <b>MAP2_TEXTURE_COORD_2</b>: If enabled, calls to #evalCoord2, #evalMesh2, and #evalPoint2 generate @a s and @a t texture coordinates.  See #map2.
        - <b>MAP2_TEXTURE_COORD_3</b>: If enabled, calls to #evalCoord2, #evalMesh2, and #evalPoint2 generate @a s, @a t, and @a r texture coordinates.  See #map2.
        - <b>MAP2_TEXTURE_COORD_4</b>: If enabled, calls to #evalCoord2, #evalMesh2, and #evalPoint2 generate @a s, @a t, @a r, and @a q texture coordinates.  See #map2.
        - <b>MAP2_VERTEX_3</b>: If enabled, calls to #evalCoord2, #evalMesh2, and #evalPoint2 generate @a x, @a y, and @a z vertex coordinates.  See #map2.
        - <b>MAP2_VERTEX_4</b>: If enabled, calls to #evalCoord2, #evalMesh2, and #evalPoint2 generate homogeneous @a x, @a y, @a z, and @a w vertex coordinates.  See #map2.
        - <b>NORMALIZE</b>: If enabled, normal vectors specified with #normal are scaled to unit length after transformation. See #normal.
        - <b>POINT_SMOOTH</b>: If enabled, draw points with proper filtering.  Otherwise, draw aliased points.  See #pointSize.
        - <b>POLYGON_OFFSET_FILL</b>: If enabled, and if the polygon is rendered in <b>FILL</b> mode, an offset is added to depth values of a polygon's fragments before the depth comparison is performed. See #polygonOffset.
        - <b>POLYGON_OFFSET_LINE</b>: If enabled, and if the polygon is rendered in <b>LINE</b> mode, an offset is added to depth values of a polygon's fragments before the depth comparison is performed. See #polygonOffset.
        - <b>POLYGON_OFFSET_POINT</b>: If enabled, an offset is added to depth values of a polygon's fragments before the depth comparison is performed, if the polygon is rendered in <b>POINT</b> mode. See #polygonOffset.
        - <b>POLYGON_SMOOTH</b>: If enabled, draw polygons with proper filtering.  Otherwise, draw aliased polygons. For correct anti-aliased polygons, an alpha buffer is needed and the polygons must be sorted front to back.
        - <b>POLYGON_STIPPLE</b>: If enabled, use the current polygon stipple pattern when rendering polygons. See #polygonStipple.
        - <b>SCISSOR_TEST</b>: If enabled, discard fragments that are outside the scissor rectangle.  See #scissor.
        - <b>SEPARATE_SPECULAR_COLOR</b>: If enabled, add the specular color after multiplying the light color by the texture color.  Otherwise, multiply them all together.  Enabling this results in proper blending of the specular highlight when texturing.  It has no effect when not texturing.
        - <b>STENCIL_TEST</b>: If enabled, do stencil testing and update the stencil buffer. See #stencilFunc and #stencilOp.
        - <b>TEXTURE_1D</b>: If enabled, one-dimensional texturing is performed (unless two-dimensional texturing is also enabled). See #texImage1D.
        - <b>TEXTURE_2D</b>: If enabled, two-dimensional texturing is performed. See #texImage2D.
        - <b>TEXTURE_GEN_Q</b>: If enabled, the @a q texture coordinate is computed using the texture generation function defined with #texGen.  Otherwise, the current @a q texture coordinate is used.  See #texGen.
        - <b>TEXTURE_GEN_R</b>: If enabled, the @a r texture coordinate is computed using the texture generation function defined with #texGen.  Otherwise, the current @a r texture coordinate is used.  See #texGen.
        - <b>TEXTURE_GEN_S</b>: If enabled, the @a s texture coordinate is computed using the texture generation function defined with #texGen.  Otherwise, the current @a s texture coordinate is used.  See #texGen.
        - <b>TEXTURE_GEN_T</b>: If enabled, the @a t texture coordinate is computed using the texture generation function defined with #texGen.  Otherwise, the current @a t texture coordinate is used.  See #texGen.
      *
      */

    void enable (GLenum cap)
    {
        switch (cap)
        {
            case COLOR_ARRAY: glEnableClientState (cap); return;
            case DEPTH_WRITEMASK: glDepthMask (TRUE); return;
            case EDGE_FLAG_ARRAY: glEnableClientState (cap); return;
            case INDEX_ARRAY: glEnableClientState (cap); return;
            case LIGHT_MODEL_LOCAL_VIEWER: glLightModeli (cap, TRUE); return;
            case LIGHT_MODEL_TWO_SIDE: glLightModeli (cap, TRUE); return;
            case NORMAL_ARRAY: glEnableClientState (cap); return;
            case SEPARATE_SPECULAR_COLOR: glLightModeli (LIGHT_MODEL_COLOR_CONTROL_EXT, SEPARATE_SPECULAR_COLOR_EXT); return;
            case TEXEL_ARRAY: glEnableClientState (cap); return;
            case VERTEX_ARRAY: glEnableClientState (cap); return;
            default: glEnable (cap);
        }
    }

    /** Enable a server-side capability and return the previous value. */
    bit enableAndGet (GLenum cap)
    {
        bit old = isEnabled (cap);

        enable (cap);
        return old;
    }

    /** Finish the primitives started with #begin. */
    void end () 
    { 
        glEnd (); 
    }

    /** Set the fog mode (mode is one of <b>LINEAR</b>, <b>EXP</b>, and <b>EXP2</b>, initially <b>EXP</b>).
        Where @a e is the value of #fogEnd, @a s is the value of #fogStart, @a z is the
        depth of a point, and @a f is the blending factor, the equation for each mode is:
  
        - <b>LINEAR</b>: f = (e - z) / (e - s)

        - <b>EXP</b>: f = exp (-d * z)

        - <b>EXP2</b>: f = exp (-(d * z) ^ 2)
  
        These are then clamped to the [0, 1] range and used to interpolate
        between transparent (0) and the fog color (1).
      */

    void fogMode (GLenum mode) { glFogi (FOG_MODE, mode); }

    /** Set the fog density for exponential mode (value must be non-negative, initially 1.) */
    void fogDensity (GLfloat value) { glFogf (FOG_DENSITY, value); }

    /** Set the calculation used for determining how far a point is from the camera.
      * This depends upon extensions; currently NV_fog_distance.
      * The setting can be:
      *
      * - <b>EYE_PLANE</b> (default): The distance is computed as the distance from the eye plane.
      *
      * - <b>EYE_RADIAL</b>: The distance is computed as the Euclidean distance from the eye.
      *
      * - <b>EYE_PLANE_ABSOLUTE</b>: As with the above, but using abs.
      */

    void fogDistanceMode (GLenum mode)
    {
        if ("NV_fog_distance" in digCommonCurrentCanvas.extensions)
            glFogi (FOG_DISTANCE_MODE_NV, mode);
    }

    /** Get the fog distance calculation. */
    GLenum fogDistanceMode ()
    {
        if ("NV_fog_distance" in digCommonCurrentCanvas.extensions)
            return getInteger (FOG_DISTANCE_MODE_NV);
        return EYE_PLANE;
    }

    /** Set the near plane for the fog in linear fog mode (initially 0.) */
    void fogStart (GLfloat value) { glFogf (FOG_START, value); }

    /** Get the near plane for the fog in linear fog mode. */
    GLfloat fogStart () { return getFloat (FOG_START); }

    /** Set the far plane for the fog in linear fog mode (initially 1.) */
    void fogEnd (GLfloat value) { glFogf (FOG_END, value); }

    /** Get the far plane for the fog in linear fog mode. */
    GLfloat fogEnd () { return getFloat (FOG_END); }

    /** Set the color index for the fog (initially 0.) */
    void fogIndex (GLint value) { glFogi (FOG_INDEX, value); }

    /** Set the fog color (initially (0, 0, 0, 0).) */
    void fogColor (GLfloat r, GLfloat g, GLfloat b) { GLfloat [4] v; v [0] = r; v [1] = g; v [2] = b; v [3] = 0; glFogfv (FOG_COLOR, v); }

    /** Set the fog color (initially (0, 0, 0, 0).) */
    void fogColor (GLfloat r, GLfloat g, GLfloat b, GLfloat a) { GLfloat [4] v; v [0] = r; v [1] = g; v [2] = b; v [3] = a; glFogfv (FOG_COLOR, v); }

    /** Set the fog color (initially (0, 0, 0, 0).) */
    void fogColor (Color c) { GLfloat [4] v; v [0] = c.r / 255.0; v [1] = c.g / 255.0; v [2] = c.b / 255.0; v [3] = c.a / 255.0; glFogfv (FOG_COLOR, v); }

    /** Get a std.math.single double of GL state. */
    GLdouble getDouble (GLenum pname) 
    { 
        GLdouble result; 
    
        glGetDoublev (pname, &result); 
        return result; 
    }

    /** Get a std.math.single integer of GL state. */
    GLint getInteger (GLenum pname) 
    { 
        GLint result; 
    
        glGetIntegerv (pname, &result); 
        return result; 
    }


    /** Set the front face parameter (CW or CCW (the default)).
      * This controls which material parameters are used as well as
      * whether this results in culling, depending upon whether
      * <b>CULL_FACE</b> is enabled.
      */

    void frontFace (GLenum mode) { glFrontFace (mode); }

    /** Get the front face parameter. */
    GLenum frontFace () { return getInteger (FRONT_FACE); }

    /** Multiply the current matrix by a perspective matrix (stolen from Mesa, thanks guys).
      * left and right are the coordinates for the vertical clipping planes.
      * bottom and top are the coordinates for the horizontal clipping planes.
      *
      * zNear and zFar are the distances to the near and far depth clipping
      * planes.  Depth buffer precision is affected by the values specified
	  * for zNear and zFar.  The greater the ratio of zFar to zNear
	  * is, the less effective the depth buffer will be at
	  * distinguishing between surfaces that are near each other.
      */
    void frustum (GLdouble left, GLdouble right,
                  GLdouble bottom, GLdouble top, 
                  GLdouble zNear, GLdouble zFar)
    {
        GLdouble x, y, a, b, c, d;
        mat4 m;

        x = (2.0 * zNear) / (right - left);
        y = (2.0 * zNear) / (top - bottom);
        a = (right + left) / (right - left);
        b = (top + bottom) / (top - bottom);
        c = -(zFar + zNear) / ( zFar - zNear);
        d = -(2.0 * zFar * zNear) / (zFar - zNear);

        m.row (0, x, 0, a, 0);
        m.row (1, 0, y, b, 0);
        m.row (2, 0, 0, c, d);
        m.row (3, 0, 0,-1, 0);

        multMatrix (m);
    }

    /** Get a std.math.single boolean of GL state. */
    bit getBoolean (GLenum pname) { GLboolean result; glGetBooleanv (pname, &result); return (bit) result; }

    /** Get a std.math.single float of GL state. */
    GLfloat getFloat (GLenum pname) { GLfloat result; glGetFloatv (pname, &result); return result; }

    /** Get a light; pass a LIGHTi value. */
    GLlight getLight (GLenum index) { return new GLlight (index); }

    /** Get a three-dimensional vector of GL state. */
    vec3 getVector3f (GLenum pname) { vec3 result; glGetFloatv (pname, result.array ()); return result; }

    /** Get the error value. */
    GLenum getError () { return glGetError (); }

    /** Get the number of channels in the color format. */
    GLint getInternalFormat (GLenum format)
    {
        switch (format)
        {
            case COLOR_INDEX:
            case RED:
            case GREEN:
            case BLUE:
            case ALPHA:
            case LUMINANCE:
            case STENCIL_INDEX:
            case DEPTH_COMPONENT:
                return 1;
            case LUMINANCE_ALPHA:
                return 2;
            case RGB:
                return 3;
            case RGBA:
                return 4;
        }
    }

private:
    double colorData (ubyte *data, GLenum type, int index)
    {
        switch (type)
        {
            case UNSIGNED_BYTE: return ((GLubyte *) data) [index] / (double) GLubyte.max;
            case BYTE: return ((GLbyte *) data) [index] / (double) GLbyte.max;
            case UNSIGNED_SHORT: return ((GLushort *) data) [index] / (double) GLushort.max;
            case SHORT: return ((GLshort *) data) [index] / (double) GLshort.max;
            case UNSIGNED_INT: return ((GLuint *) data) [index] / (real) GLuint.max;
            case INT: return ((GLint *) data) [index] / (real) GLint.max;
            case FLOAT: return ((GLfloat *) data) [index];
            case DOUBLE: return ((GLdouble *) data) [index];
        }
    }

public:
    /** Read a color from data using the given format and type. */
    Color colorFromData (ubyte *data, GLenum format, GLenum type)
    {
        double a = colorData (data, type, 0);

        switch (format)
        {
            case COLOR_INDEX: throw new Error ("colorFromData: Cannot get color from index.");
            case RED: return SColor1 (a, 0, 0);
            case GREEN: return SColor1 (0, a, 0);
            case BLUE: return SColor1 (0, 0, a);
            case ALPHA: return SColor1 (0, 0, 0, a);
            case LUMINANCE: return SColor1 (a, a, a);
            case STENCIL_INDEX: return SColor1 (a, a, a);
            case DEPTH_COMPONENT: return SColor1 (a, a, a);
            case LUMINANCE_ALPHA: return SColor1 (a, a, a, colorData (data, type, 1));
            case RGB: return SColor1 (a, colorData (data, type, 1), colorData (data, type, 2));
            case RGBA: return SColor1 (a, colorData (data, type, 1), colorData (data, type, 2), colorData (data, type, 3));
        }
    }

private:
    void toColor (ubyte *data, GLenum type, int index, ubyte value)
    {
        const int max = 255;

        switch (type)
        {
            case UNSIGNED_BYTE: ((GLubyte *) data) [index] = value; break;
            case BYTE: ((GLbyte *) data) [index] = value * GLbyte.max / max; break;
            case UNSIGNED_SHORT: ((GLushort *) data) [index] = (real) value * GLushort.max / max; break;
            case SHORT: ((GLshort *) data) [index] = (real) value * GLshort.max / max; break;
            case UNSIGNED_INT: ((GLuint *) data) [index] = (real) value * GLuint.max / max; break;
            case INT: ((GLint *) data) [index] = (real) value * GLint.max / max; break;
            case FLOAT: ((GLfloat *) data) [index] = (real) value / max; break;
            case DOUBLE: ((GLdouble *) data) [index] = (real) value / max; break;
        }
    }
public:

    /** Write a color in data using the given format and type. */
    void colorToData (ubyte *data, GLenum format, GLenum type, Color color, Color background)
    {
        Color blank = background.blend (color);

        switch (format)
        {
            case COLOR_INDEX: throw new Error ("colorToData: Cannot set color from index.");
            case RED: toColor (data, type, 0, blank.r); break;
            case GREEN: toColor (data, type, 0, blank.g); break;
            case BLUE: toColor (data, type, 0, blank.b); break;
            case ALPHA: toColor (data, type, 0, color.a); break;
            case LUMINANCE: toColor (data, type, 0, blank.intensity ()); break;
            case STENCIL_INDEX: toColor (data, type, 0, color.r); break;
            case DEPTH_COMPONENT: toColor (data, type, 0, color.r); break;
            case LUMINANCE_ALPHA: toColor (data, type, 0, color.intensity ()); toColor (data, type, 0, color.a); break;
            case RGB: toColor (data, type, 0, color.r); toColor (data, type, 1, color.g); toColor (data, type, 2, color.b); break;
            case RGBA: toColor (data, type, 0, color.r); toColor (data, type, 1, color.g); toColor (data, type, 2, color.b); toColor (data, type, 3, color.a); break;
        }
    }

    /** Write a color in data using the given format and type and a black background. */
    void colorToData (ubyte *data, GLenum format, GLenum type, Color color)
    {
        colorToData (data, format, type, color, Color.Black);
    }

    /** Return a string describing the GL connection (name is one of VENDOR, RENDERER, VERSION, or EXTENSIONS).
        - <b>VENDOR</b>:
          Returns the company responsible for this
          GL implementation.  This name does not
          change from release to release.

        - <b>RENDERER</b>:
          Returns the name of the renderer.  This
          name is typically specific to a
          particular configuration of a hardware
          platform.  It does not change from
          release to release.

        - <b>VERSION</b>:
          Returns a	version	or release number.

        - <b>EXTENSIONS</b>:
          Returns a	space-separated	list of supported extensions to GL.
      */

    char [] getString (GLenum name)
    {
        char *result = (char *) glGetString (name);

        if (result == null)
            throwError ();
        return result [0 .. std.string.strlen (result)];
    }

    /** Get the number of bytes in a given type (one of <b>UNSIGNED_BYTE</b>, <b>BYTE</b>,
      * <b>UNSIGNED_SHORT</b>, <b>SHORT</b>, <b>UNSIGNED_INT</b>, <b>INT</b>,
      * or <b>FLOAT</b>). */
    GLint getTypeSize (GLenum type)
    {
        switch (type)
        {
            case UNSIGNED_BYTE: return GLubyte.size;
            case BYTE: return GLbyte.size;
            case UNSIGNED_SHORT: return GLushort.size;
            case SHORT: return GLshort.size;
            case UNSIGNED_INT: return GLuint.size;
            case INT: return GLint.size;
            case FLOAT: return GLfloat.size;
            case DOUBLE: return GLdouble.size;
        }
    }

    /** Set polygon smooth hinting; value is <b>FASTEST</b>, <b>NICEST</b>, or <b>DONT_CARE</b>. */
    void hintPolygonSmooth (GLenum value) { glHint (POLYGON_SMOOTH, value); }

    /** Test whether a server-side capability is enabled. */
    bit isEnabled (GLenum cap)
    {
        switch (cap)
        {
            case DEPTH_WRITEMASK: return getBoolean (cap);
            case LIGHT_MODEL_LOCAL_VIEWER: return getBoolean (cap);
            case LIGHT_MODEL_TWO_SIDE: return getBoolean (cap);
            case SEPARATE_SPECULAR_COLOR: return (bit) getInteger (LIGHT_MODEL_COLOR_CONTROL_EXT) == SEPARATE_SPECULAR_COLOR;
            default: return (bit) glIsEnabled (cap);
        }
    }

    /** Set the ambient color of a light object. */
    void lightAmbient (GLenum light, GLfloat r, GLfloat g, GLfloat b) { GLfloat [4] v; v [0] = r; v [1] = g; v [2] = b; v [3] = 1; glLightfv (light, AMBIENT, v); }

    /** Set the position of a light object. */
    void lightPosition (GLenum light, GLfloat x, GLfloat y, GLfloat z) { GLfloat [4] v; v [0] = x; v [1] = y; v [2] = z; v [3] = 0; glLightfv (light, POSITION, v); }

    /** Get the position of a light object. */
    vec3 lightPosition (GLenum light) { GLfloat [4] v; glGetLightfv (light, POSITION, v); return vec3.create (v [0], v [1], v [2]); }

    /** Set the diffuse color of a light object. */
    void lightDiffuse (GLenum light, GLfloat r, GLfloat g, GLfloat b) { GLfloat [4] v; v [0] = r; v [1] = g; v [2] = b; v [3] = 1; glLightfv (light, DIFFUSE, v); }

    /** Set the diffuse color of a light object. */
    void lightDiffuse (GLenum light, Color c) { GLfloat [4] v; v [0] = c.r / 255.0; v [1] = c.g / 255.0; v [2] = c.b / 255.0; v [3] = c.a / 255.0; glLightfv (light, DIFFUSE, v); }

    /** Get the diffuse color of the light. */
    Color lightDiffuse (GLenum light) { GLfloat [4] c; glGetLightfv (light, DIFFUSE, c); return SColor1 (c [0], c [1], c [2], c [3]); }

    /** Set the specular color of a light object. */
    void lightSpecular (GLenum light, GLfloat r, GLfloat g, GLfloat b) { GLfloat [4] v; v [0] = r; v [1] = g; v [2] = b; v [3] = 1; glLightfv (light, SPECULAR, v); }

    /** Set the three attenuation parameters of the light simultaneously. */
    void lightAttenuation (GLenum light, GLfloat constant, GLfloat linear, GLfloat quadratic)
    {
        lightConstantAttenuation (light, constant);
        lightLinearAttenuation (light, linear);
        lightQuadraticAttenuation (light, quadratic);
    }

    /** Set the constant attenuation of the light. */
    void lightConstantAttenuation (GLenum light, GLfloat value) { glLightfv (light, CONSTANT_ATTENUATION, &value); }

    /** Set the linear attenuation of the light. */
    void lightLinearAttenuation (GLenum light, GLfloat value) { glLightfv (light, LINEAR_ATTENUATION, &value); }

    /** Set the quadratic attenuation of the light. */
    void lightQuadraticAttenuation (GLenum light, GLfloat value) { glLightfv (light, QUADRATIC_ATTENUATION, &value); }

    /** Set the normal of the spotlight. */
    void lightSpotDirection (GLenum light, GLfloat x, GLfloat y, GLfloat z) { GLfloat [3] v; v [0] = x; v [1] = y; v [2] = z; glLightfv (light, SPOT_DIRECTION, v); }

    /** Set the normal of the spotlight. */
    void lightSpotDirection (GLenum light, vec3 v) { glLightfv (light, SPOT_DIRECTION, v.array ()); }

    /** Set the cutoff angle of the spotlight (0 to 90 degrees, or 180 degrees (default) for a normal light). */
    void lightSpotCutoff (GLenum light, GLfloat angle) { glLightfv (light, SPOT_CUTOFF, &angle); }

    /** Set the intensity distribution of the spotlight (0 for diffuse, 128 for hard-edged). */
    void lightSpotExponent (GLenum light, GLfloat factor) { glLightfv (light, SPOT_EXPONENT, &factor); }

    /** Set the global ambient color (defaults to (0.2, 0.2, 0.2, 1.0)). */
    void lightModelAmbient (GLfloat r, GLfloat g, GLfloat b) { GLfloat [4] v; v [0] = r; v [1] = g; v [2] = b; v [3] = 1; glLightModelfv (LIGHT_MODEL_AMBIENT, v); }

    /** Set the global ambient color (defaults to (0.2, 0.2, 0.2, 1.0)). */
    void lightModelAmbient (GLfloat r, GLfloat g, GLfloat b, GLfloat a) { GLfloat [4] v; v [0] = r; v [1] = g; v [2] = b; v [3] = a; glLightModelfv (LIGHT_MODEL_AMBIENT, v); }

    /** Set the global ambient color. */
    void lightModelAmbient (Color c) { GLfloat [4] v; v [0] = c.r / 255.0; v [1] = c.g / 255.0; v [2] = c.b / 255.0; v [3] = c.a / 255.0; glLightModelfv (LIGHT_MODEL_AMBIENT, v); }

    /** Set the width in pixels of the lines. */
    void lineWidth (GLfloat value) { glLineWidth (value); }

    /** Get the current line width. */
    GLfloat lineWidth () { return getFloat (LINE_WIDTH); }

    /** Get the minimum line width difference for a visible change. */
    GLfloat lineWidthGranularity () { return getFloat (LINE_WIDTH_GRANULARITY); }

    /** Get the maximum line width. */
    GLfloat lineWidthMax () { GLfloat [2] v; glGetFloatv (LINE_WIDTH_RANGE, v); return v [1]; }

    /** Get the minimum line width. */
    GLfloat lineWidthMin () { GLfloat [2] v; glGetFloatv (LINE_WIDTH_RANGE, v); return v [0]; }

    /** Create a new display list object. */
    GLlist listNew () { return new GLlist (); }

    /** Create a display list name. */
    GLuint listCreate () { return glGenLists (1); }

    /** Delete a display list. */
    void listDelete (GLuint list) { glDeleteLists (list, 1); }

    /** Start compilation on the list. */
    void listCompile (GLuint list) { glNewList (list, COMPILE); }

    /** Start compilation on the list and execute at the same time. */
    void listCompileAndExecute (GLuint list) { glNewList (list, COMPILE_AND_EXECUTE); }

    /** Finish compilation. */
    void listCompileEnd () { glEndList (); }

    /** Run the display list. */
    void listCall (GLuint list) { glCallList (list); }

    /** Set the current matrix to identity. */
    void loadIdentity () { glLoadIdentity (); }

    /** Set the given matrix to identity without changing the current matrix. */
    void loadIdentity (GLenum mode) { GLenum old = matrixMode (); matrixMode (mode); loadIdentity (); matrixMode (old); }

    /** Specify a logical pixel operation for color rendering.
      * This logic operation blends the final fragment color with
      * the color buffer value when #enable (<b>COLOR_LOGIC_OP</b>).
      * A warning: this is more than likely not accelerated on your
      * hardware.  In the following, @a s is a bit from the source color
      * @a d is a bit from the frame buffer color.  The
      * valid values for @a opcode and their operations are:
      *
      * - <b>CLEAR</b>: 0
      * - <b>SET</b>: 1
      * - <b>COPY</b>: s
      * - <b>COPY_INVERTED</b>: ~s
      * - <b>NOOP</b>: d
      * - <b>INVERT</b>: ~d
      * - <b>AND</b>: s & d
      * - <b>NAND</b>: ~(s & d)
      * - <b>OR</b>: s | d
      * - <b>NOR</b>: ~(s | d)
      * - <b>XOR</b>: s ^ d
      * - <b>EQUIV</b>: ~(s ^ d)
      * - <b>AND_REVERSE</b>: s & ~d
      * - <b>AND_INVERTED</b>: ~s & d
      * - <b>OR_REVERSE</b>: s | ~d
      * - <b>OR_INVERTED</b>: ~s | d
      */

    void logicOp (GLenum opcode) { glLogicOp (opcode); }

    /** Set the ambient material color. */
    void materialAmbient (GLenum face, GLfloat r, GLfloat g, GLfloat b) { GLfloat [4] v; v [0] = r; v [1] = g; v [2] = b; v [3] = 1; glMaterialfv (face, AMBIENT, v); }

    /** Set the ambient material color. */
    void materialAmbient (GLenum face, GLfloat r, GLfloat g, GLfloat b, GLfloat a) { GLfloat [4] v; v [0] = r; v [1] = g; v [2] = b; v [3] = a; glMaterialfv (face, AMBIENT, v); }

    /** Set the ambient material color. */
    void materialAmbient (GLenum face, Color c) { GLfloat [4] v; v [0] = c.r / 255.0; v [1] = c.g / 255.0; v [2] = c.b / 255.0; v [3] = c.a / 255.0; glMaterialfv (face, AMBIENT, v); }

    /** Get the ambient color of the material. */
    Color materialAmbient (GLenum face) { GLfloat [4] v; glGetMaterialfv (face, AMBIENT, v); return AColor (v [0] * 255.0, v [1] * 255.0, v [2] * 255.0, v [3] * 255.0); }

    /** Set the ambient and diffuse material colors to the same value. */
    void materialAmbientAndDiffuse (GLenum face, GLfloat r, GLfloat g, GLfloat b) { GLfloat [4] v; v [0] = r; v [1] = g; v [2] = b; v [3] = 1; glMaterialfv (face, AMBIENT_AND_DIFFUSE, v); }

    /** Set the ambient and diffuse material colors. */
    void materialAmbientAndDiffuse (GLenum face, GLfloat r, GLfloat g, GLfloat b, GLfloat a) { GLfloat [4] v; v [0] = r; v [1] = g; v [2] = b; v [3] = a; glMaterialfv (face, AMBIENT_AND_DIFFUSE, v); }

    /** Set the ambient and diffuse material colors. */
    void materialAmbientAndDiffuse (GLenum face, Color c) { GLfloat [4] v; v [0] = c.r / 255.0; v [1] = c.g / 255.0; v [2] = c.b / 255.0; v [3] = c.a / 255.0; glMaterialfv (face, AMBIENT_AND_DIFFUSE, v); }

    /** Set the diffuse material color. */
    void materialDiffuse (GLenum face, GLfloat r, GLfloat g, GLfloat b) { materialDiffuse (face, r, g, b, 1.0); }

    /** Set the diffuse material color. */
    void materialDiffuse (GLenum face, GLfloat r, GLfloat g, GLfloat b, GLfloat a)
    {
        GLfloat [4] v; 
        
        v [0] = r; v [1] = g; v [2] = b; v [3] = a;
        
        glMaterialfv (face, DIFFUSE, v);
    }

    /** Set the diffuse material color. */
    void materialDiffuse (GLenum face, Color c) { GLfloat [4] v; v [0] = c.r / 255.0; v [1] = c.g / 255.0; v [2] = c.b / 255.0; v [3] = c.a / 255.0; glMaterialfv (face, DIFFUSE, v); }

    /** Get the diffuse color of the material. */
    Color materialDiffuse (GLenum face) { GLfloat [4] v; glGetMaterialfv (face, DIFFUSE, v); return AColor (v [0] * 255.0, v [1] * 255.0, v [2] * 255.0, v [3] * 255.0); }

    /** Set the material emission color; this is not multiplied against anything, so has a literal effect on color. */
    void materialEmission (GLenum face, GLfloat r, GLfloat g, GLfloat b) { GLfloat [4] v; v [0] = r; v [1] = g; v [2] = b; v [3] = 1; glMaterialfv (face, EMISSION, v); }

    /** Set the material emission color. */
    void materialEmission (GLenum face, GLfloat r, GLfloat g, GLfloat b, GLfloat a) { GLfloat [4] v; v [0] = r; v [1] = g; v [2] = b; v [3] = a; glMaterialfv (face, EMISSION, v); }

    /** Set the material emission color. */
    void materialEmission (GLenum face, Color c) { GLfloat [4] v; v [0] = c.r / 255.0; v [1] = c.g / 255.0; v [2] = c.b / 255.0; v [3] = c.a / 255.0; glMaterialfv (face, EMISSION, v); }

    /** Get the emission color of the material. */
    Color materialEmission (GLenum face) { GLfloat [4] v; glGetMaterialfv (face, EMISSION, v); return AColor (v [0] * 255.0, v [1] * 255.0, v [2] * 255.0, v [3] * 255.0); }

    /** Set the shininess of the material. */
    void materialShininess (GLenum face, GLfloat value) { glMaterialfv (face, SHININESS, &value); }

    /** Set the specular color of the material. */
    void materialSpecular (GLenum face, GLfloat r, GLfloat g, GLfloat b) { GLfloat [4] v; v [0] = r; v [1] = g; v [2] = b; v [3] = 1; glMaterialfv (face, SPECULAR, v); }

    /** Set the specular color of the material. */
    void materialSpecular (GLenum face, GLfloat r, GLfloat g, GLfloat b, GLfloat a) { GLfloat [4] v; v [0] = r; v [1] = g; v [2] = b; v [3] = a; glMaterialfv (face, SPECULAR, v); }

    /** Set the specular color of the material. */
    void materialSpecular (GLenum face, Color c) { GLfloat [4] v; v [0] = c.r / 255.0; v [1] = c.g / 255.0; v [2] = c.b / 255.0; v [3] = c.a / 255.0; glMaterialfv (face, SPECULAR, v); }

    /** Get the specular color of the material. */
    Color materialSpecular (GLenum face) { GLfloat [4] v; glGetMaterialfv (face, SPECULAR, v); return AColor (v [0] * 255.0, v [1] * 255.0, v [2] * 255.0, v [3] * 255.0); }

    /** Set the current matrix mode (mode is one of PROJECTION, MODELVIEW, or TEXTURE.) */
    void matrixMode (GLenum mode) { glMatrixMode (mode); }

    /** Get the current matrix mode. */
    GLenum matrixMode () { GLint [1] value; glGetIntegerv (MATRIX_MODE, value); return value [0]; }

    /** Get the number of lights supported by the GL; this must be 8. */
    GLint maxLights () { return getInteger (MAX_LIGHTS); }

    /** Get the modelview matrix. */
    mat4 modelviewMatrix ()
    {
        mat4 result;

        glGetFloatv (MODELVIEW_MATRIX, result.array ());
        return result;
    }

    /** Multiply the current matrix by this matrix. */
    void multMatrix (GLdouble [16] matrix) { glMultMatrixd (matrix); }

    /** Multiply the current matrix by a matrix. */
    void multMatrix (mat4 matrix) { glMultMatrixf (matrix.array ()); }

    /** Set the current normal vector.  If !#isEnabled (<b>NORMALIZE</b>),
      * the default, then the vector is not normalized after translation.
      */

    void normal (GLfloat x, GLfloat y, GLfloat z) { glNormal3f (x, y, z); }

    /** Set the current normal vector. */
    void normal (GLfloat [3] v) { glNormal3fv (v); }

    /** Set the current normal vector. */
    void normal (vec3 v) { glNormal3f (v.x, v.y, v.z); }

    /** Get the current normal vector. */
    vec3 normal () { return getVector3f (CURRENT_NORMAL); }

    /** Set the normal array; use #enable (<b>NORMAL_ARRAY</b>) to enable it.
      * #a stride should be zero to indicate 12 (the size of the array) or
      * the number of bytes between each entry.
      */

    void normalArray (vec3 *array, int stride)
    {
        glNormalPointer (FLOAT, stride, array);
    }

    /** Sets up a perspective projection matrix (stolen from Mesa, thanks guys). */
    void perspective (GLdouble fovy, GLdouble aspect, GLdouble zNear, GLdouble zFar)
    {
        GLdouble xmin, xmax, ymin, ymax;

        ymax = zNear * std.math.tan (fovy * std.math.PI / 360.0);
        ymin = -ymax;
        xmin = ymin * aspect;
        xmax = ymax * aspect;

        frustum (xmin, xmax, ymin, ymax, zNear, zFar);
    }

    /** Specify the diameter of rasterized points (initially 1). */
    void pointSize (GLfloat value) { glPointSize (value); }

    /** Get the point size. */
    GLfloat pointSize () { return getFloat (POINT_SIZE); }

    /** Set a polygon rasterization mode (@a face is <b>BACK</b>,
        <b>FRONT</b>, or <b>FRONT_AND_BACK</b>; @a mode is <b>POINT</b>,
        <b>LINE</b>, or <b>FILL</b> (the default)).

        This controls the interpretation of polygons for rasterization.

        - <b>POINT</b>: Polygon vertices that are marked as the start of a boundary edge are drawn as points.  Point attributes such as #pointSize and #enable (<b>POINT_SMOOTH</b>) control the rasterization of the points.  Polygon rasterization attributes other than polygonMode have no effect.
        - <b>LINE</b>: Boundary edges of the polygon are drawn as line segments.  They are treated as connected line segments for line stippling; the line stipple counter and pattern are not reset between segments (see #lineStipple).  Line attributes such as #lineWidth and #enable (<b>LINE_SMOOTH</b> control the rasterization of the lines.  Polygon rasterization attributes other than #polygonMode have no effect.
        - <b>FILL</b>: The interior of the polygon is filled. Polygon attributes such as #polygonStipple and #enable (<b>POLYGON_SMOOTH</b>) control the rasterization of the polygon.
      */

    void polygonMode (GLenum face, GLenum mode) { glPolygonMode (face, mode); }

    /** Get the polygon rasterization mode for a face (@a face is <b>BACK</b> or <b>FRONT</b>). */
    GLenum polygonMode (GLenum face) { GLint [2] value; glGetIntegerv (POLYGON_MODE, value); if (face == FRONT) return value [0]; if (face == BACK) return value [1]; throw new GLError (INVALID_VALUE); }

    /** Set the scale and units used to calculate depth values when 
      * #enable (<b>POLYGON_OFFSET_mode</b>), where mode is <b>FILL</b>,
      * <b>LINE</b>, or <b>POINT</b>.
      */

    void polygonOffset (GLfloat factor, GLfloat units) { glPolygonOffset (factor, units); }

    /** Pop a matrix off the current matrix's stack. */
    void popMatrix () { glPopMatrix (); }

    /** Get the projection matrix. */
    mat4 projectionMatrix ()
    {
        mat4 result;

        glGetFloatv (PROJECTION_MATRIX, result.array ());
        return result;
    }

    /** Push a matrix onto the current matrix's stack. */
    void pushMatrix () { glPushMatrix (); }

    /** Read a block of pixels from the frame buffer, returning the
      * result.  x and y are the coordinates (a y of zero is the bottom
      * line), width and height are the dimensions to read, and format
      * is the buffer to read from, and can be <b>COLOR_INDEX</b>, 
      * <b>STENCIL_INDEX</b>, <b>DEPTH_COMPONENT</b>, <b>RED</b>,
      * <b>GREEN</b>, <b>BLUE</b>, <b>ALPHA</b>, <b>RGB</b>, <b>RGBA</b>, 
      * <b>LUMINANCE</b>, and <b>LUMINANCE_ALPHA</b>.  The result is stored
      * in data.  Values for pixels that are not in range are undefined.
      */

    void readPixels (GLint x, GLint y, GLsizei width, GLsizei height, GLenum format, out GLubyte [] data)
    {
        int pitch = width * getInternalFormat (format);

        data = new GLubyte [pitch * height];
        glReadPixels (x, y, width, height, format, UNSIGNED_BYTE, data);
    }

    /** Read pixels into an array of bytes. */
    void readPixels (GLint x, GLint y, GLsizei width, GLsizei height, GLenum format, out GLbyte [] data)
    {
        int pitch = width * getInternalFormat (format);

        data = new GLbyte [pitch * height];
        glReadPixels (x, y, width, height, format, BYTE, data);
    }

    /** Read pixels into an array of unsigned shorts. */
    void readPixels (GLint x, GLint y, GLsizei width, GLsizei height, GLenum format, out GLushort [] data)
    {
        int pitch = width * getInternalFormat (format);

        data = new GLushort [pitch * height];
        glReadPixels (x, y, width, height, format, UNSIGNED_SHORT, data);
    }

    /** Read pixels into an array of shorts. */
    void readPixels (GLint x, GLint y, GLsizei width, GLsizei height, GLenum format, out GLshort [] data)
    {
        int pitch = width * getInternalFormat (format);

        data = new GLshort [pitch * height];
        glReadPixels (x, y, width, height, format, SHORT, data);
    }

    /** Read pixels into an array of unsigned ints. */
    void readPixels (GLint x, GLint y, GLsizei width, GLsizei height, GLenum format, out GLuint [] data)
    {
        int pitch = width * getInternalFormat (format);

        data = new GLuint [pitch * height];
        glReadPixels (x, y, width, height, format, UNSIGNED_INT, data);
    }

    /** Read pixels into an array of ints. */
    void readPixels (GLint x, GLint y, GLsizei width, GLsizei height, GLenum format, out GLint [] data)
    {
        int pitch = width * getInternalFormat (format);

        data = new GLint [pitch * height];
        glReadPixels (x, y, width, height, format, INT, data);
    }

    /** Read pixels into an array of floats. */
    void readPixels (GLint x, GLint y, GLsizei width, GLsizei height, GLenum format, out GLfloat [] data)
    {
        int pitch = width * getInternalFormat (format);

        data = new GLfloat [pitch * height];
        glReadPixels (x, y, width, height, format, FLOAT, data);
    }

    /** Rotate the current matrix around an angle. */
    void rotate (GLfloat angle, GLfloat x, GLfloat y, GLfloat z) { glRotatef (angle, x, y, z); }

    /** Rotate the current matrix around (0, 0, 1). */
    void rotate (GLfloat angle) { glRotatef (angle, 0, 0, 1); }

    /** Scale the current matrix by a factor. */
    void scale (GLfloat factor) { glScalef (factor, factor, factor); }

    /** Scale the current matrix by an amount. */
    void scale (GLfloat x, GLfloat y, GLfloat z) { glScalef (x, y, z); }

    /** Scale the current matrix by a vector. */
    void scale (vec3 v) { glScalef (v.x, v.y, v.z); }

    /** Setup the matrices to give a perspective mode. */
    void setupPerspective (GLfloat fov, GLfloat aspect, GLfloat near, GLfloat far)
    {
        GLenum mode = matrixMode ();

        loadIdentity (MODELVIEW);

        matrixMode (PROJECTION);
        loadIdentity ();
        perspective (fov, aspect, near, far);
        matrixMode (mode);
    }

    /** Render a sphere (stolen from Mesa, thanks guys). */
    void sphere (GLdouble radius, GLint slices, GLint stacks)
    {
        GLfloat rho, drho, theta, dtheta;
        GLfloat x, y, z;
        GLfloat s, t, ds, dt;
        GLint i, j, imin, imax;
        GLboolean normals, texcoords;
        GLfloat nsign;
        
        texcoords = TRUE; /* whether to create texcoords */
        normals = TRUE; /* whether to create normals */
        nsign = 1.0; /* -1 for inside */
        
        drho = std.math.PI / (GLfloat) stacks;
        dtheta = 2.0 * std.math.PI / (GLfloat) slices;
        
        /* texturing: s goes from 0.0/0.25/0.5/0.75/1.0 at +y/+x/-y/-x/+y axis */
        /* t goes from -1.0/+1.0 at z = -radius/+radius (linear along longitudes) */
        /* cannot use triangle fan on texturing (s coord. at top/bottom tip varies) */
        
        if (!texcoords)
        {
            /* draw +Z end as a triangle fan */
            glBegin (TRIANGLE_FAN);
            glNormal3f (0.0, 0.0, 1.0);
            glVertex3f (0.0, 0.0, nsign * radius);
            for (j = 0; j <= slices; j++)
            {
                theta = (j == slices) ? 0.0 : j * dtheta;
                x = -std.math.sin(theta) * std.math.sin(drho);
                y = std.math.cos(theta) * std.math.sin(drho);
                z = nsign * std.math.cos(drho);
                if (normals)
                    glNormal3f (x * nsign, y * nsign, z * nsign);
                glVertex3f (x * radius, y * radius, z * radius);
            }
            glEnd ();
        }
        
        ds = 1.0 / slices;
        dt = 1.0 / stacks;
        t = 1.0;			/* because loop now runs from 0 */
        if (texcoords)
            imin = 0, imax = stacks;
        else
            imin = 1, imax = stacks - 1;
        
        /* draw intermediate stacks as quad strips */
        for (i = imin; i < imax; i ++)
        {
            rho = i * drho;
            glBegin (QUAD_STRIP);
            s = 0.0;
            for (j = 0; j <= slices; j ++)
            {
                theta = (j == slices) ? 0.0 : j * dtheta;
                x = -std.math.sin(theta) * std.math.sin(rho);
                y = std.math.cos(theta) * std.math.sin(rho);
                z = nsign * std.math.cos(rho);
                if (normals)
                    glNormal3f (x * nsign, y * nsign, z * nsign);
                if (texcoords)
                    glTexCoord2f (s, t);
                glVertex3f(x * radius, y * radius, z * radius);
                x = -std.math.sin(theta) * std.math.sin(rho + drho);
                y = std.math.cos(theta) * std.math.sin(rho + drho);
                z = nsign * std.math.cos(rho + drho);
                if (normals)
                    glNormal3f(x * nsign, y * nsign, z * nsign);
                if (texcoords)
                    glTexCoord2f (s, t - dt);
                s += ds;
                glVertex3f (x * radius, y * radius, z * radius);
            }
            glEnd();
            t -= dt;
        }
        
        if (!texcoords)
        {
            /* draw -Z end as a triangle fan */
            glBegin (TRIANGLE_FAN);
            glNormal3f (0.0, 0.0, -1.0);
            glVertex3f (0.0, 0.0, -radius * nsign);
            rho = std.math.PI - drho;
            s = 1.0;
            t = dt;
            for (j = slices; j >= 0; j--)
            {
                theta = (j == slices) ? 0.0 : j * dtheta;
                x = -std.math.sin (theta) * std.math.sin (rho);
                y = std.math.cos (theta) * std.math.sin (rho);
                z = nsign * std.math.cos (rho);
                if (normals)
                    glNormal3f (x * nsign, y * nsign, z * nsign);
                s -= ds;
                glVertex3f (x * radius, y * radius, z * radius);
            }
            glEnd ();
        }
    }

    /** Set the function and reference value for stencil testing; @a func is
        <b>NEVER</b>, <b>LESS</b>, <b>LEQUAL</b>, <b>GREATER</b>,
        <b>GEQUAL</b>, <b>EQUAL</b>, <b>NOTEQUAL</b>, and <b>ALWAYS</b>
        (initial value); @a ref is the reference value for the stencil
        test; while @a mask is the AND mask used for reading and writing
        the stencil buffer.

        Stencil testing is enabled with #enable (<b>STENCIL_TEST</b>),
        and the effect on the stencil buffer from the test is controlled
        with #stencilOp.  @a func is one of:

        - <b>NEVER</b>: Always fails.
        - <b>LESS</b>: Passes if (@a ref & @a mask) < (@a stencil & @a mask).
        - <b>LEQUAL</b>: Passes if (@a ref & @a mask) < (@a stencil & @a mask).
        - <b>GREATER</b>: Passes if (@a ref & @a mask) > (@a stencil & @a mask).
        - <b>GEQUAL</b>: Passes if (@a ref & @a mask) > (@a stencil & @a mask).
        - <b>EQUAL</b>: Passes if (@a ref & @a mask) = (@a stencil & @a mask).
        - <b>NOTEQUAL</b>: Passes if (@a ref & @a mask) /  (@a stencil & @a mask).
        - <b>ALWAYS</b>: Always passes.
      */

    void stencilFunc (GLenum func, GLint ref, GLuint mask) { glStencilFunc (func, ref, mask); }

    /** Set the function and reference value for stencil testing; mask is taken to be ~0. */
    void stencilFunc (GLenum func, GLint ref) { glStencilFunc (func, ref, ~0); }

    /** Set the effects stencil testing have on the stencil buffer; each is
        either <b>KEEP</b>, <b>ZERO</b>, <b>REPLACE</b>, <b>INCR</b>,
        <b>DECR</b>, or <b>INVERT</b>.

        Stencil testing is enabled with #enable (<b>STENCIL_TEST</b>),
        and the test is controlled with #stencilFunc.

        The @a fail action is taken when the stencil test fails.  The
        @a zfail action is taken when the stencil test passes, but the
        depth test fails.  @a zpass is taken when both the stencil test
        and the depth test pass, or when the stencil test passes and
        there is no depth buffer or it is not enabled.  The actions are:

        - <b>KEEP</b>: Keeps the current value.
        - <b>ZERO</b>: Sets the stencil buffer value to 0.
        - <b>REPLACE</b>: Sets the stencil buffer value to @a ref, as specified by #stencilFunc.
        - <b>INCR</b>: Increments the current stencil buffer value.  Clamps to the maximum representable unsigned value.
        - <b>DECR</b>: Decrements the current stencil buffer value.  Clamps to 0.
        - <b>INVERT</b>: Bitwise inverts the current stencil buffer value.
      */

    void stencilOp (GLenum fail, GLenum zfail, GLenum zpass) { glStencilOp (fail, zfail, zpass); }
  
    /** Enable or disable the capability depending upon the value. */
    void store (GLenum cap, GLenum value)
    {
        if (value)
            enable (cap);
        else
            disable (cap);
    }

    /** Set the current texture coordinate. */
    void texel (vec2 v)
    {
        glTexCoord2f (v.x, v.y);
    }

    /** Set the current texture coordinate. */
    void texel (float x, float y)
    {
        glTexCoord2f (x, y);
    }

    /** Set the current texture coordinate of the given texture unit, in the range
      * <b>TEXTURE0...</b>#textureUnitCount.
      */

    void texel (GLenum unit, vec2 v)
    {
        digCommonCurrentCanvas.glMultiTexCoord2f (unit, v.x, v.y);
    }

    /** Set the texel array; use #enable (<b>TEXEL_ARRAY</b>) to enable it.
      * #a stride should be zero to indicate 8 (the size of the cell) or
      * the number of bytes between each entry.
      */

    void texelArray (vec2 *array, int stride)
    {
        glTexCoordPointer (2, FLOAT, stride, array);
    }

    /** Set the texel array; use #enable (<b>TEXEL_ARRAY</b>) to enable it.
      * #a stride should be zero to indicate 12 (the size of the cell) or the
      * number of bytes between each entry.
      */

    void texelArray (vec3 *array, int stride)
    {
        glTexCoordPointer (3, FLOAT, stride, array);
    }

    /** Specify texture coordinate generation mode (@a coord is <b>S</b>,
        <b>T</b>, <b>R</b>, or <b>Q</b>; @a value is <b>OBJECT_LINEAR</b>,
        <b>EYE_LINEAR</b>, or <b>SPHERE_MAP</b>).

        Texture coordinate generation is enabled using <b>TEXTURE_GEN_S</b>,
        <b>TEXTURE_GEN_T</b>, <b>TEXTURE_GEN_R</b>, and <b>TEXTURE_GEN_Q</b>.

        - <b>OBJECT_LINEAR</b>: The texture coordinate is the distance of the point from the plane specified in #textureGenObjectPlane.
        - <b>EYE_LINEAR</b>: TODO.
        - <b>SPHERE_MAP</b>: The texture coordinate is selected based on a sphere map.
      */

    void texelGenMode (GLenum coord, GLenum param)
    {
        glTexGeni (coord, TEXTURE_GEN_MODE, param);
    }

    /** Indicate the plane to use for #texelGenMode (<b>OBJECT_LINEAR</b>). */
    void texelGenObjectPlane (GLenum coord, float distance, float x, float y, float z)
    {
        GLfloat [4] v;

        v [0] = distance;
        v [1] = x;
        v [2] = y;
        v [3] = z;
        glTexGenfv (coord, OBJECT_PLANE, v);
    }

    /** Create a new texture object. */
    GLtexture textureNew () { return new GLtexture (); }

    /** Generate a std.math.single texture name. */
    GLuint textureCreate () { GLuint value; glGenTextures (1, &value); return value; }

    /** Generate a number of textures, storing in the array. */
    void textureList (GLuint [] list) { glGenTextures (list.length, list); }

    /** Bind a named texture to the <b>TEXTURE_2D</b> target. */
    void textureBind (GLuint texture) { glBindTexture (TEXTURE_2D, texture); }

    /** Copy a rectangle of the color buffer into the texture,
      * replacing any data which was in there before.  level is the
      * mipmapping level to read from.  format is the texture format
      * to store and can be <b>ALPHA</b>, <b>LUMINANCE</b>, 
      * <b>LUMINANCE_ALPHA</b>, <b>INTENSITY</b>, <b>RGB</b>, <b>RGBA</b>,
      * or a number of more obscure modes.  x and y are the starting
      * coordinates for the color buffer rectangle; a y of zero is the
      * bottom row of the buffer.  width and height are the dimensions
      * of the rectangle and must be powers of two.
      */

    void textureCopyImage (GLint level, GLenum format, GLint x, GLint y, GLint width, GLint height)
    {
        glCopyTexImage2D (TEXTURE_2D, level, format, x, y, width, height, 0);
    }

    /** Delete a texture. */
    void textureDelete (GLuint texture) { glDeleteTextures (1, &texture); }

    /** Delete a list of textures. */
    void textureDelete (GLuint [] textures) { glDeleteTextures (textures.length, textures); }

    /** Assign the texture environment mode.  This can be <b>MODULATE</b>,
      * <b>DECAL</b>, <b>BLEND</b>, <b>REPLACE</b>, or <b>COMBINE</b>.
      * The default is <b>MODULATE</b>.
      */

    void textureEnvMode (GLenum param)
    {
        glTexEnvi (TEXTURE_ENV, TEXTURE_ENV_MODE, param);
    }

    /** Assign the color combination function for #textureEnvMode (<b>COMBINE</b>)
      * texture environment mode.
      * This can be <b>REPLACE</b> (arg0), <b>MODULATE</b> (arg0 * arg1),
      * <b>ADD</b> (arg0 + arg1), <b>ADD_SIGNED</b> (arg0 + arg1 - 0.5), 
      * <b>INTERPOLATE</b> (arg0 * arg2 + arg1 * (1 - arg2)), or
      * <b>SUBTRACT</b> (arg0 - arg1).  arg0, arg1, and arg2 are assigned through two
      * factors: #textureEnvSourceRGB and #textureEnvOperandRGB, where 0 is 0, 1, or 2.
      * The default is <b>MODULATE</b>.
      */

    void textureEnvCombineRGB (GLenum param)
    {
        glTexEnvi (TEXTURE_ENV, COMBINE_RGB, param);
    }

    /** Assign source arguments for the #textureEnvCombineRGB function.  Arg
      * is the argument to modify (0, 1, or 2), and param is the value.  This can be
      * <b>TEXTURE</b> (read from the texture), <b>CONSTANT</b> (use a
      * constant color), <b>PRIMARY_COLOR</b> (use the main #color or the
      * light color computation if #enable (<b>LIGHTING</b>)), <b>PREVIOUS</b>
      * (the color from the previous texturing unit, see #textureUnit).  The defaults
      * are <b>TEXTURE</b>, <b>PREVIOUS</b>, and <b>CONSTANT</b> for
      * the three arguments.
      */

    void textureEnvSourceRGB (int arg, GLenum param)
    {
        glTexEnvi (TEXTURE_ENV, SOURCE0_RGB + arg, param);
    }

    /** Return the source argument for the #textureEnvCombineRGB function. */

    GLenum textureEnvSourceRGB (int arg)
    {
        GLint param;

        glGetTexEnviv (TEXTURE_ENV, SOURCE0_RGB + arg, &param);
        return param;
    }

    /** Load a two-dimensional texture image into the currently bound texture, simple version.
      * @a width and @a height are the dimensions of the texture in pixels.
      * @a format is the provided channels and can be <b>COLOR_INDEX</b>,
      * <b>RED</b>, <b>GREEN</b>, <b>BLUE</b>, <b>ALPHA</b>,
      * <b>RGB</b>, <b>RGBA</b>, <b>LUMINANCE</b>, and <b>LUMINANCE_ALPHA</b>.
      * @a type is the data type and can be <b>UNSIGNED_BYTE</b> (GLubyte),
      * <b>BYTE</b> (GLbyte), <b>BITMAP</b> (compressed bits, @a format must be <b>COLOR_INDEX</b>),
      * <b>UNSIGNED_SHORT</b> (GLushort), <b>SHORT</b> (GLshort),
      * <b>UNSIGNED_INT</b> (GLuint), <b>INT</b> (GLint), and
      * <b>FLOAT</b> (GLfloat).  @a pixels is the array to read from;
      * if null, the texture dimensions are set but are not cleared.
      *
      * The image is loaded from the bottom up, so the first pixel
      * in the @a pixels array is the bottom-left one.
      *
      * If the width and height are not powers of two (1, 2, 4, 8, 16, 32,
      * 64, 128, 256, 512, 1024, 2048, 4096, or 8192), the image is first
      * resized to the nearest demarcation.  Then it is checked whether
      * it fits, and is progressively halved in size until it does.  Mipmaps
      * are automatically generated, either manually or by extension.
      */

    void textureImage (GLsizei width, GLsizei height, GLenum format, GLenum type, GLvoid *pixels)
    {
        gluBuild2DMipmaps (TEXTURE_2D, getInternalFormat (format), width, height, format, type, pixels);
    }

    /** Specify a two-dimensional texture image, including level.  The @a level
      * is the mipmapping level this image is inserted into.  0 is the base
      * image; n is the nth mipmap reduction.
      */

    void textureImage (GLint level, GLsizei width, GLsizei height, GLenum format, GLenum type, GLvoid *pixels)
    {
        GLenum target = TEXTURE_2D;
        GLint internalformat = getInternalFormat (format);
        GLint border = 0;

        textureImage (target, level, internalformat, width, height, border, format, type, pixels);
    }

    /** Specify a two-dimensional texture image, complex version.
      * See OpenGL documentation for glTexImage2D for details.
      */

    void textureImage (GLenum target, GLint level, GLint internalformat,
                       GLsizei width, GLsizei height, GLint border, GLenum format,
                       GLenum type, GLvoid *pixels)
    {
        glTexImage2D (target, level, internalformat, width, height, border, format, type, pixels);
    }

    /** Read the texture image from GL state, storing it in pixels. */
    void textureImage (GLint level, GLenum format, out GLubyte [] pixels)
    {
        int pitch = textureWidth (level) * getInternalFormat (format);

        pixels = new GLubyte [pitch * textureHeight (level)];
        glGetTexImage (TEXTURE_2D, level, format, UNSIGNED_BYTE, pixels);
    }

    /** Set the magnification filter of the currently-bound texture (either
        <b>NEAREST</b> or <b>LINEAR</b>).

        The magnification filter is used when the pixel is smaller than a
        texture element.  It can be one of:

        - <b>NEAREST</b>: Returns the value of the texture element that is nearest (in Manhattan distance) to the center of the pixel being textured.
        - <b>LINEAR</b>: Returns the weighted average of the four texture elements that are closest to the center of the pixel being textured. These can include border texture elements, depending on the values of #textureWrapS and #textureWrapT, and on the exact mapping.
      */

    void textureMagFilter (GLenum value) { glTexParameteri (TEXTURE_2D, TEXTURE_MAG_FILTER, value); }

    /** Set the minification filter of the currently-bound texture (one of
        <b>NEAREST</b>, <b>LINEAR</b>, <b>NEAREST_MIPMAP_NEAREST</b>,
        <b>LINEAR_MIPMAP_NEAREST</b>, <b>NEAREST_MIPMAP_LINEAR</b>, or
        <b>LINEAR_MIPMAP_LINEAR</b>).
      
        The minification filter is used when the pixel is larger than a
        texture element.  It can be one of:

        - <b>NEAREST</b>: Returns the value of the texture element that is nearest (in Manhattan distance) to the center of the pixel being textured.
        - <b>LINEAR</b>: Returns the weighted average of the four texture elements that are closest to the center of the pixel being textured. These can include border texture elements, depending on the values of #textureWrapS and #textureWrapT, and on the exact mapping.
        - <b>NEAREST_MIPMAP_NEAREST</b>: Chooses the mipmap that most closely matches the size of the pixel being textured and uses the <b>NEAREST</b> criterion (the texture element nearest to the center of the pixel) to produce a texture value.
        - <b>LINEAR_MIPMAP_NEAREST</b>: Chooses the mipmap that most closely matches the size of the pixel being textured and uses the <b>LINEAR</b> criterion (a weighted average of the four texture elements that are closest to the center of the pixel) to produce a texture value.
        - <b>NEAREST_MIPMAP_LINEAR</b>: Chooses the two mipmaps that most closely match the size of the pixel being textured and uses the <b>NEAREST</b> criterion (the texture element nearest to the center of the pixel) to produce a texture value from each mipmap. The final texture value is a weighted average of those two values.
        - <b>LINEAR_MIPMAP_LINEAR</b>: Chooses the two mipmaps that most closely match the size of the pixel being textured and uses the <b>LINEAR</b> criterion (a weighted average of the four texture elements that are closest to the center of the pixel) to produce a texture value from each mipmap.  The final texture value is a weighted average of those two values.     

        <b>LINEAR</b> is often called bilinear filtering, while
        <b>LINEAR_MIPMAP_LINEAR</b> is often called trilinear filtering.
        <b>NEAREST</b> and <b>LINEAR</b> are often slower than the
        other forms due to what's called cache thrashing: too much access to
        widely spaced sections of memory.
      */

    void textureMinFilter (GLenum value) { glTexParameteri (TEXTURE_2D, TEXTURE_MIN_FILTER, value); }

    /** Specify the currently bound texture's residence priority, in the range [0, 1], default is 1. */
    void texturePriority (GLfloat value) { glTexParameterf (TEXTURE_2D, TEXTURE_PRIORITY, value); }

    /** Retrieve the currently bound texture's residence priority. */
    GLfloat texturePriority () { GLfloat value; glGetTexParameterfv (TEXTURE_2D, TEXTURE_PRIORITY, &value); return value; }

    /** Get the width of the currently-bound texture. */
    GLint textureWidth () { return textureWidth (0); }

    /** Get the width of the currently-bound texture at this level. */
    GLint textureWidth (GLint level) { GLint value; glGetTexLevelParameteriv (TEXTURE_2D, level, TEXTURE_WIDTH, &value); return value; }

    /** Get the height of the currently-bound texture. */
    GLint textureHeight () { return textureHeight (0); }

    /** Get the height of the currently-bound texture at this level. */
    GLint textureHeight (GLint level) { GLint value; glGetTexLevelParameteriv (TEXTURE_2D, level, TEXTURE_HEIGHT, &value); return value; }

    /** Find out whether automatic mipmap generation is available (whether @a texGenerateMipmap can be used). */
    GLboolean textureCanGenerateMipmap () { return "GL_SGIS_generate_mipmap" in digCommonCurrentCanvas.extensions; }

    /** Specify whether to automatically create mipmaps; this is not available on all cards.
      * It must be enabled before uploading the texture and returns whether it
      * was successful.
      */

    GLboolean textureGenerateMipmap (GLboolean value)
    {
        if (!("GL_SGIS_generate_mipmap" in digCommonCurrentCanvas.extensions))
            return FALSE;
        glTexParameteri (TEXTURE_2D, GENERATE_MIPMAP_SGIS, value);
        return TRUE;
    }

    /** Get the texture matrix. */
    mat4 textureMatrix ()
    {
        mat4 result;

        glGetFloatv (TEXTURE_MATRIX, result.array ());
        return result;
    }

    /** Change the active texturing unit, which is in the range
      * <b>TEXTURE0...</b>#textureUnitCount.
      */

    void textureUnit (GLenum unit)
    {
        digCommonCurrentCanvas.glActiveTexture (unit);
        digCommonCurrentCanvas.glClientActiveTexture (unit);
    }

    /** Return the number of multitexturing units available.  You can switch
      * the current texture unit using #textureUnit and can enable or disable
      * individual units using #enable (<b>TEXTURE0...32</b>).
      */

    GLint textureUnitCount ()
    {
        return digCommonCurrentCanvas.textureUnitCount;
    }

    /** Sets the wrap parameter for	texture	coordinate @a s
        to either <b>CLAMP</b> or <b>REPEAT</b>.  <b>CLAMP</b> causes
        @a s coordinates to be clamped	to the range [0, 1] and
        is useful for preventing wrapping artifacts	when
		mapping a std.math.single image onto	an object. <b>REPEAT</b>
		causes the integer part of the @a s coordinate to be
		ignored; the GL uses only the fractional part,
		thereby creating a repeating pattern. Border
		texture elements are accessed only if wrapping is
		set to <b>CLAMP</b>.  Initially, @a textureWrapS is
		set to <b>REPEAT</b>.
      */

    void textureWrapS (GLenum value)
    {
        glTexParameteri (TEXTURE_2D, TEXTURE_WRAP_S, value);
    }

    /** Get the wrap parameter for texture coordinate @a s. */
    GLenum textureWrapS ()
    {
        GLint value;

        glGetTexParameteriv (TEXTURE_2D, TEXTURE_WRAP_S, &value);
        return value;
    }

    /** Sets the wrap parameter for texture coordinate @a t
        to either <b>CLAMP</b> or <b>REPEAT</b>.  <b>CLAMP</b> causes
        @a t coordinates to be clamped	to the range [0, 1] and
        is useful for preventing wrapping artifacts	when
		mapping a std.math.single image onto	an object. <b>REPEAT</b>
		causes the integer part of the @a t coordinate to be
		ignored; the GL uses only the fractional part,
		thereby creating a repeating pattern. Border
		texture elements are accessed only if wrapping is
		set to <b>CLAMP</b>.  Initially, @a textureWrapT is
		set to <b>REPEAT</b>.
      */

    void textureWrapT (GLenum value)
    {
        glTexParameteri (TEXTURE_2D, TEXTURE_WRAP_T, value);
    }

    /** Get the wrap parameter for texture coordinate @a t. */
    GLenum textureWrapT ()
    {
        GLint value;

        glGetTexParameteriv (TEXTURE_2D, TEXTURE_WRAP_T, &value);
        return value;
    }

    /** Retrieve and throw the last error, or return if there is none. */
    void throwError ()
    {
        GLenum error;

        if (error != NO_ERROR)
            throw new GLError (error);
    }

    /** Translate the current matrix. */
    void translate (GLfloat x, GLfloat y) { glTranslatef (x, y, 0); }

    /** Translate the current matrix. */
    void translate (GLfloat x, GLfloat y, GLfloat z) { glTranslatef (x, y, z); }

    /** Translate the current matrix. */
    void translate (vec3 vec) { glTranslatef (vec.x, vec.y, vec.z); }

    /** Translate the current matrix. */
    void translate (vec2 vec) { glTranslatef (vec.x, vec.y, 0); }

    /** Transform screen coordinates into object-space coordinates. */
    vec3 unproject (vec2 point) { return unproject (vec3.create (point.x, point.y, 0)); }

    /** Transform screen coordinates into object-space coordinates. */
    vec3 unproject (vec3 point)
    {
        GLdouble winx = point.x, winy = point.y, winz = point.z;
        mat4 model, proj, m, a;
        GLint [4] viewport;
        vec4 i, o;

        model = modelviewMatrix ();
        proj = projectionMatrix ();
        glGetIntegerv (VIEWPORT, viewport);

        i.x = (winx - viewport [0]) * 2 / viewport [2] - 1.0;
        i.y = (winy - viewport [1]) * 2 / viewport [3] - 1.0;
        i.z = 2 * winz - 1.0;
        i.w = 1.0;

        a = proj * model;
        a = a.invert ();
        o = m * i;

        return vec3.create (o.x / o.w, o.y / o.w, o.z / o.w);
    }

    /** Store a two-dimensional vertex. */
    void vertex (GLfloat x, GLfloat y) { glVertex2f (x, y); }

    /** Store a three-dimensional vertex. */
    void vertex (GLfloat x, GLfloat y, GLfloat z) { glVertex3f (x, y, z); }

    /** Store a four-dimensional vertex. */
    void vertex (GLfloat x, GLfloat y, GLfloat z, GLfloat w) { glVertex4f (x, y, z, w); }

    /** Store a three-dimensional vertex. */
    void vertex (vec3 v) { glVertex3f (v.x, v.y, v.z); }

    /** Set the vertex array; use #enable (<b>VERTEX_ARRAY</b>) to enable it.
      * #a stride should be zero to indicate 8 (the size of the array) or
      * the number of bytes between each entry.
      */

    void vertexArray (vec3 *array, int stride)
    {
        glVertexPointer (3, FLOAT, stride, array);
    }

    /** Set the viewport area. */
    void viewport (GLint x, GLint y, GLsizei width, GLsizei height) { glViewport (x, y, width, height); }

    /** Get the viewport area. */
    void viewportArea (out GLint x, out GLint y, out GLsizei width, out GLsizei height)
    {
        GLint [4] v;

        glGetIntegerv (VIEWPORT, v);
        x = v [0];
        y = v [1];
        width = v [2];
        height = v [3];
    }

/+
#ifndef DOXYGEN_SHOULD_SKIP_THIS
+/

    import net.BurtonRadons.dig.common.glEnums;

    /* Compute ceiling of integer quotient of A divided by B: */
    static int CEILING (int a, int b) { return a % b == 0 ? (a / b) : (a / b) + 1; }

    /* Stolen from Mesa, thanks guys! */
    GLint gluScaleImage (GLenum format,
                         GLsizei widthin, GLsizei heightin,
                         GLenum typein, void *datain,
                         GLsizei widthout, GLsizei heightout,
                         GLenum typeout, void *dataout)
    {
        GLint components, i, j, k;
        GLfloat [] tempin, tempout;
        GLfloat sx, sy;
        GLint unpackrowlength, unpackalignment, unpackskiprows, unpackskippixels;
        GLint packrowlength, packalignment, packskiprows, packskippixels;
        GLint sizein, sizeout;
        GLint rowstride, rowlen;

        components = getInternalFormat (format); /* Determine number of components per pixel */
        sizein = getTypeSize (typein);
        sizeout = getTypeSize (typeout);
        
        /* Get glPixelStore state */
        glGetIntegerv (UNPACK_ROW_LENGTH, &unpackrowlength);
        glGetIntegerv (UNPACK_ALIGNMENT, &unpackalignment);
        glGetIntegerv (UNPACK_SKIP_ROWS, &unpackskiprows);
        glGetIntegerv (UNPACK_SKIP_PIXELS, &unpackskippixels);
        glGetIntegerv (PACK_ROW_LENGTH, &packrowlength);
        glGetIntegerv (PACK_ALIGNMENT, &packalignment);
        glGetIntegerv (PACK_SKIP_ROWS, &packskiprows);
        glGetIntegerv (PACK_SKIP_PIXELS, &packskippixels);
        
        /* Allocate storage for intermediate images */
        tempin = new GLfloat [widthin * heightin * components];
        tempout = new GLfloat [widthout * heightout * components];
        
        /* Unpack the pixel data and convert to floating point */
        
        if (unpackrowlength > 0)
            rowlen = unpackrowlength;
        else
            rowlen = widthin;
        
        if (sizein >= unpackalignment)
            rowstride = components * rowlen;
        else
            rowstride = unpackalignment / sizein * CEILING (components * rowlen * sizein, unpackalignment);
        
        switch (typein)
        {
            case UNSIGNED_BYTE:
                k = 0;
                for (i = 0; i < heightin; i++)
                {
                    GLubyte *ubptr = (GLubyte *) datain
                        + i * rowstride
                        + unpackskiprows * rowstride + unpackskippixels * components;
                    for (j = 0; j < widthin * components; j++)
                        tempin[k++] = (GLfloat) *ubptr++;
                }
                break;
            case BYTE:
                k = 0;
                for (i = 0; i < heightin; i++)
                {
                    GLbyte *bptr = (GLbyte *) datain
                        + i * rowstride
                        + unpackskiprows * rowstride + unpackskippixels * components;
                    for (j = 0; j < widthin * components; j++)
                        tempin[k++] = (GLfloat) * bptr++;
                }
                break;
            case UNSIGNED_SHORT:
                k = 0;
                for (i = 0; i < heightin; i++)
                {
                    GLushort *usptr = (GLushort *) datain
                        + i * rowstride
                        + unpackskiprows * rowstride + unpackskippixels * components;
                    for (j = 0; j < widthin * components; j++)
                        tempin[k++] = (GLfloat) * usptr++;
                }
                break;
            case SHORT:
                k = 0;
                for (i = 0; i < heightin; i++)
                {
                    GLshort *sptr = (GLshort *) datain
                        + i * rowstride
                        + unpackskiprows * rowstride + unpackskippixels * components;
                    for (j = 0; j < widthin * components; j++)
                        tempin[k++] = (GLfloat) * sptr++;
                }
                break;
            case UNSIGNED_INT:
                k = 0;
                for (i = 0; i < heightin; i++)
                {
                    GLuint *uiptr = (GLuint *) datain
                        + i * rowstride
                        + unpackskiprows * rowstride + unpackskippixels * components;
                    for (j = 0; j < widthin * components; j++)
                        tempin[k++] = (GLfloat) * uiptr++;
                }
                break;
            case INT:
                k = 0;
                for (i = 0; i < heightin; i++)
                {
                    GLint *iptr = (GLint *) datain
                        + i * rowstride
                        + unpackskiprows * rowstride + unpackskippixels * components;
                    for (j = 0; j < widthin * components; j++)
                        tempin[k++] = (GLfloat) * iptr++;
                }
                break;
            case FLOAT:
                k = 0;
                for (i = 0; i < heightin; i++)
                {
                    GLfloat *fptr = (GLfloat *) datain
                        + i * rowstride
                        + unpackskiprows * rowstride + unpackskippixels * components;
                    for (j = 0; j < widthin * components; j++)
                        tempin[k++] = *fptr++;
                }
                break;
            default:
                throw new Error ("Invalid type enumeration");
        }
        
        
        /* Scale the image! */
        
        if (widthout > 1)
            sx = (GLfloat) (widthin - 1) / (GLfloat) (widthout - 1);
        else
            sx = (GLfloat) (widthin - 1);
        if (heightout > 1)
            sy = (GLfloat) (heightin - 1) / (GLfloat) (heightout - 1);
        else
            sy = (GLfloat) (heightin - 1);
        
        version (_POINT_SAMPLE)
        {
            for (i = 0; i < heightout; i++)
            {
                GLint ii = i * sy;
                for (j = 0; j < widthout; j++)
                {
                    GLint jj = j * sx;
                
                    GLfloat *src = tempin + (ii * widthin + jj) * components;
                    GLfloat *dst = tempout + (i * widthout + j) * components;
                
                    for (k = 0; k < components; k++)
                        *dst++ = *src++;
                }
            }
        }
        else
        {
            if (sx < 1.0 && sy < 1.0)
            {
                /* magnify both width and height:  use weighted sample of 4 pixels */
                GLint i0, i1, j0, j1;
                GLfloat alpha, beta;
                GLfloat *src00, src01, src10, src11;
                GLfloat s1, s2;
                GLfloat *dst;
            
                for (i = 0; i < heightout; i++)
                {
                    i0 = i * sy;
                    i1 = i0 + 1;
                    if (i1 >= heightin)
                        i1 = heightin - 1;
                    /*	 i1 = (i+1) * sy - EPSILON;*/
                    alpha = i * sy - i0;
                    for (j = 0; j < widthout; j++)
                    {
                        j0 = j * sx;
                        j1 = j0 + 1;
                        if (j1 >= widthin)
                            j1 = widthin - 1;
                        /*	    j1 = (j+1) * sx - EPSILON; */
                        beta = j * sx - j0;
                    
                        /* compute weighted average of pixels in rect (i0,j0)-(i1,j1) */
                        src00 = &tempin [(i0 * widthin + j0) * components];
                        src01 = &tempin [(i0 * widthin + j1) * components];
                        src10 = &tempin [(i1 * widthin + j0) * components];
                        src11 = &tempin [(i1 * widthin + j1) * components];
                    
                        dst = &tempout [(i * widthout + j) * components];
                    
                        for (k = 0; k < components; k++)
                        {
                            s1 = *src00++ * (1.0 - beta) + *src01++ * beta;
                            s2 = *src10++ * (1.0 - beta) + *src11++ * beta;
                            *dst++ = s1 * (1.0 - alpha) + s2 * alpha;
                        }
                    }
                }
            }
            else
            {
                /* shrink width and/or height:  use an unweighted box filter */
                GLint i0, i1;
                GLint j0, j1;
                GLint ii, jj;
                GLfloat sum;
                GLfloat *dst;
            
                for (i = 0; i < heightout; i++)
                {
                    i0 = i * sy;
                    i1 = i0 + 1;
                    if (i1 >= heightin)
                        i1 = heightin - 1;
                    /*	 i1 = (i+1) * sy - EPSILON; */
                    for (j = 0; j < widthout; j++)
                    {
                        j0 = j * sx;
                        j1 = j0 + 1;
                        if (j1 >= widthin)
                            j1 = widthin - 1;
                        /*	    j1 = (j+1) * sx - EPSILON; */
                    
                        dst = &tempout [(i * widthout + j) * components];
                    
                        /* compute average of pixels in the rectangle (i0,j0)-(i1,j1) */
                        for (k = 0; k < components; k++)
                        {
                            sum = 0.0;
                            for (ii = i0; ii <= i1; ii++)
                            {
                                for (jj = j0; jj <= j1; jj++)
                                sum += tempin [(ii * widthin + jj) * components + k];
                            }
                            sum /= (j1 - j0 + 1) * (i1 - i0 + 1);
                            *dst++ = sum;
                        }
                    }
                }
            }
        }       
        
        /*
        * Return output image
        */
        
        if (packrowlength > 0)
            rowlen = packrowlength;
        else
            rowlen = widthout;

        if (sizeout >= packalignment)
            rowstride = components * rowlen;
        else
            rowstride = packalignment / sizeout
                * CEILING(components * rowlen * sizeout, packalignment);
        
        switch (typeout)
        {
            case UNSIGNED_BYTE:
                k = 0;
                for (i = 0; i < heightout; i++)
                {
                    GLubyte *ubptr = (GLubyte *) dataout
                        + i * rowstride
                        + packskiprows * rowstride + packskippixels * components;
                    for (j = 0; j < widthout * components; j++)
                        *ubptr++ = (GLubyte) tempout[k++];
                }
                break;

            case BYTE:
                k = 0;
                for (i = 0; i < heightout; i++) {
                    GLbyte *bptr = (GLbyte *) dataout
                        + i * rowstride
                        + packskiprows * rowstride + packskippixels * components;
                    for (j = 0; j < widthout * components; j++) {
                        *bptr++ = (GLbyte) tempout[k++];
                    }
                }
                break;
            case UNSIGNED_SHORT:
                k = 0;
                for (i = 0; i < heightout; i++) {
                    GLushort *usptr = (GLushort *) dataout
                        + i * rowstride
                        + packskiprows * rowstride + packskippixels * components;
                    for (j = 0; j < widthout * components; j++) {
                        *usptr++ = (GLushort) tempout[k++];
                    }
                }
                break;
            case SHORT:
                k = 0;
                for (i = 0; i < heightout; i++) {
                    GLshort *sptr = (GLshort *) dataout
                        + i * rowstride
                        + packskiprows * rowstride + packskippixels * components;
                    for (j = 0; j < widthout * components; j++) {
                        *sptr++ = (GLshort) tempout[k++];
                    }
                }
                break;
            case UNSIGNED_INT:
                k = 0;
                for (i = 0; i < heightout; i++) {
                    GLuint *uiptr = (GLuint *) dataout
                        + i * rowstride
                        + packskiprows * rowstride + packskippixels * components;
                    for (j = 0; j < widthout * components; j++) {
                        *uiptr++ = (GLuint) tempout[k++];
                    }
                }
                break;
            case INT:
                k = 0;
                for (i = 0; i < heightout; i++) {
                    GLint *iptr = (GLint *) dataout
                        + i * rowstride
                        + packskiprows * rowstride + packskippixels * components;
                    for (j = 0; j < widthout * components; j++) {
                        *iptr++ = (GLint) tempout[k++];
                    }
                }
                break;
            case FLOAT:
                k = 0;
                for (i = 0; i < heightout; i++) {
                    GLfloat *fptr = (GLfloat *) dataout
                        + i * rowstride
                        + packskiprows * rowstride + packskippixels * components;
                    for (j = 0; j < widthout * components; j++) {
                        *fptr++ = tempout[k++];
                    }
                }
                break;
            default:
                throw new Error ("Invalid type enumeration");
        }
        
done:        
        /* free temporary image storage */
        delete tempin;
        delete tempout;
        
        return 0;
    }

    /* Return the largest k such that 2^k <= n. */
    static GLint ilog2(GLint n)
    {
        GLint k;
    
        if (n <= 0)
            return 0;
        for (k = 0; n >>= 1; k++)
            {}
        return k;
    }


    /* Find the value nearest to n which is also a power of two. */
    static GLint round2 (GLint n)
    {
        GLint m;
    
        for (m = 1; m < n; m *= 2) {}
    
        /* m>=n */
        if (m - n <= n - m / 2)
            return m;
        else
            return m / 2;
    }

    /*
     * Given an pixel format and datatype, return the number of bytes to
     * store one pixel.
     */
    GLint bytes_per_pixel (GLenum format, GLenum type)
    {
        return getInternalFormat (format) * getTypeSize (type);
    }

    /* WARNING: This function isn't finished and has never been tested!!!! */
    GLint gluBuild1DMipmaps (GLenum target, GLint components,
        GLsizei width, GLenum format, GLenum type, void *data)
    {
        GLubyte *texture;
        GLint levels, max_levels;
        GLint new_width, max_width;
        GLint i, j, k, l;
        
        if (width < 1)
            throw new Error ("Width below one; that's wack, foo!");
        
        glGetIntegerv(MAX_TEXTURE_SIZE, &max_width);
        max_levels = ilog2(max_width) + 1;
        
        /* Compute how many mipmap images to make */
        levels = ilog2(width) + 1;
        if (levels > max_levels) {
            levels = max_levels;
        }
        
        new_width = 1 << (levels - 1);
        
        texture = new GLubyte [new_width * components];
        
        if (width != new_width)
        {
            /* initial rescaling */
            switch (type)
            {
                case UNSIGNED_BYTE:
                {
                    GLubyte *ub_data = (GLubyte *) data;
                    for (i = 0; i < new_width; i++) {
                        j = i * width / new_width;
                        for (k = 0; k < components; k++) {
                            texture[i * components + k] = ub_data[j * components + k];
                        }
                    }
                    break;
                }
                default:
                    /* Not implemented */
                    throw new Error ("gluBuild1DMipmaps only handles type UNSIGNED_BYTE");
            }
        }
        
        /* generate and load mipmap images */
        for (l = 0; l < levels; l++)
        {
            glTexImage1D (TEXTURE_1D, l, components, new_width, 0, format, UNSIGNED_BYTE, texture);
            
            /* Scale image down to 1/2 size */
            new_width = new_width / 2;
            for (i = 0; i < new_width; i++)
            {
                for (k = 0; k < components; k++)
                {
                    GLint sample1, sample2;
                    sample1 = (GLint) texture[i * 2 * components + k];
                    sample2 = (GLint) texture[(i * 2 + 1) * components + k];
                    texture[i * components + k] = (GLubyte) ((sample1 + sample2) / 2);
                }
            }
        }
    
        delete texture;
    
        return 0;
    }


    GLint
        gluBuild2DMipmaps (GLenum target, GLint components,
                           GLsizei width, GLsizei height, GLenum format,
                           GLenum type, void *data)
    {
        GLint w, h, maxsize;
        void *image, newimage;
        GLint neww, newh, level, bpp;
        int error;
        GLboolean done;
        GLint retval = 0;
        GLint unpackrowlength, unpackalignment, unpackskiprows, unpackskippixels;
        GLint packrowlength, packalignment, packskiprows, packskippixels;
        GLboolean autoMipmap;
        
        if (width < 1 || height < 1)
            throw new Error ("Width or height below one; that's wack, foo!");
        
        glGetIntegerv(MAX_TEXTURE_SIZE, &maxsize);
        
        w = round2(width);
        if (w > maxsize)
            w = maxsize;
        
        h = round2(height);
        if (h > maxsize)
            h = maxsize;
        
        autoMipmap = textureGenerateMipmap (TRUE);
        bpp = bytes_per_pixel(format, type);
        
        /* Get current glPixelStore values */
        glGetIntegerv(UNPACK_ROW_LENGTH, &unpackrowlength);
        glGetIntegerv(UNPACK_ALIGNMENT, &unpackalignment);
        glGetIntegerv(UNPACK_SKIP_ROWS, &unpackskiprows);
        glGetIntegerv(UNPACK_SKIP_PIXELS, &unpackskippixels);
        glGetIntegerv(PACK_ROW_LENGTH, &packrowlength);
        glGetIntegerv(PACK_ALIGNMENT, &packalignment);
        glGetIntegerv(PACK_SKIP_ROWS, &packskiprows);
        glGetIntegerv(PACK_SKIP_PIXELS, &packskippixels);
        
        /* set pixel packing */
        glPixelStorei(PACK_ROW_LENGTH, 0);
        glPixelStorei(PACK_ALIGNMENT, 1);
        glPixelStorei(PACK_SKIP_ROWS, 0);
        glPixelStorei(PACK_SKIP_PIXELS, 0);
        
        done = FALSE;
        
        if (w != width || h != height)
        {
            /* must rescale image to get "top" mipmap texture image */
            image = new byte [(w + 4) * h * bpp];
            error = gluScaleImage (format, width, height, type, data, w, h, type, image);
            if (error)
            {
                retval = error;
                done = TRUE;
            }
        }
        else
            image = (void *) data;
        
        level = 0;
        while (!done)
        {
            if (image != data)
            {
                /* set pixel unpacking */
                glPixelStorei (UNPACK_ROW_LENGTH, 0);
                glPixelStorei (UNPACK_ALIGNMENT, 1);
                glPixelStorei (UNPACK_SKIP_ROWS, 0);
                glPixelStorei (UNPACK_SKIP_PIXELS, 0);
            }
            
            glTexImage2D (target, level, components, w, h, 0, format, type, image);
            if (autoMipmap)
                break;
            
            if (w == 1 && h == 1)
                break;
            
            neww = (w < 2) ? 1 : w / 2;
            newh = (h < 2) ? 1 : h / 2;
            newimage = new byte [(neww + 4) * newh * bpp];
            
            error = gluScaleImage (format, w, h, type, image, neww, newh, type, newimage);
            if (error)
            {
                retval = error;
                done = TRUE;
            }
            
            //if (image != data)
              //  delete image;
            image = newimage;
            
            w = neww;
            h = newh;
            level++;
        }

        //if (image != data)
        //    delete image;
        
        /* Restore original glPixelStore state */
        glPixelStorei(UNPACK_ROW_LENGTH, unpackrowlength);
        glPixelStorei(UNPACK_ALIGNMENT, unpackalignment);
        glPixelStorei(UNPACK_SKIP_ROWS, unpackskiprows);
        glPixelStorei(UNPACK_SKIP_PIXELS, unpackskippixels);
        glPixelStorei(PACK_ROW_LENGTH, packrowlength);
        glPixelStorei(PACK_ALIGNMENT, packalignment);
        glPixelStorei(PACK_SKIP_ROWS, packskiprows);
        glPixelStorei(PACK_SKIP_PIXELS, packskippixels);
        
        return retval;
    }

/+
#endif
+/
}

GL gl;
